/* #------------------------------------------------------------------------#
   |                                                                        |
   |   RISCIPX.H                                                            |
   |                                                                        |
   |   RiscIPX external interface.                                          |
   |                                                                        |
   |   Copyright 1998, Frank A. Vorstenbosch                                |
   |                                                                        |
   #------------------------------------------------------------------------# */

#include "mbuf.h"
#include "lists.h"

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Opaque structures.                                                   |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct Route;

#define FIRSTPORT       0x4000
#define LASTPORT        0x7fff


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Constants.                                                           |
   |                                                                        |
   +------------------------------------------------------------------------+ */

/* well-known sockets */
#define IPX_SKT_XNSRIP        0x0001
#define IPX_SKT_XNSECHO       0x0002
#define IPX_SKT_XNSERROR      0x0003
#define IPX_SKT_NCP           0x0451
#define IPX_SKT_SAP           0x0452
#define IPX_SKT_RIP           0x0453
#define IPX_SKT_NETBIOS       0x0455
#define IPX_SKT_DIAGNOSTICS   0x0456
#define IPX_SKT_NWSERIALNUM   0x0457
#define IPX_SKT_NLSP          0x9000

/* packet types */
#define IPX_PT_RIP   1        /* Routing Information Packet (RIP) */
#define IPX_PT_ECHO  2        /* Echo Protocol Packet             */
#define IPX_PT_ERROR 3        /* Error Handler Packet             */
#define IPX_PT_PEP   4        /* Packet Exchange Packet (PEP)     */
#define IPX_PT_SPP   5        /* Sequenced Packet Protocol (SPP)  */
#define IPX_PT_USER  16       /* User packets -- actually 16..31  */
#define IPX_PT_NCP   17       /* NetWare Core Protocol (NCP)      */

#define IPX_PT_IPX IPX_PT_PEP

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   IPX address.                                                         |
   |                                                                        |
   +------------------------------------------------------------------------+ */

#define IPX_NODE_LEN    6
struct Address
{  long Network;
   char Node[IPX_NODE_LEN];
   unsigned short Port;
};


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   More opaque structures.                                              |
   |                                                                        |
   +------------------------------------------------------------------------+ */

#ifdef _NODEF_RISCIPX_
   struct RiscIPX;
   struct Socket;
#else
   struct RiscIPX
   {  struct mbctl mbctl;
   };
   struct Socket
   {  NODE;
      struct Address Connection;
   };
#endif


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   IPX header.                                                          |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct UnpackedHeader
{  unsigned short CheckSum,
                  Length;
   /* char TransportControl  -- used as hop count, set to 0 on transmit */
   char PacketType;
   struct Address Destination,
                  Source;
};

#define PACKETHEADERSIZE (2+2+1+1+2*(4+6+2)) /* 30 bytes */

/* #------------------------------------------------------------------------#
   |                                                                        |
   |   Function prototypes.                                                 |
   |                                                                        |
   #------------------------------------------------------------------------# */

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Sockets and transmit/receive functions.                              |
   |                                                                        |
   +------------------------------------------------------------------------+ */

/* ----- Create and destroy sockets ----------------------------------------- */

extern struct Socket *ipx_socket(register struct RiscIPX *ri);
extern int ipx_close(struct Socket *sk,register struct RiscIPX *ri);
int ipx_recv_handler(struct Socket *sk,
                     int (*Handler)(struct mbuf *,struct UnpackedHeader *,void *),
                     void *Data,register struct RiscIPX *ri);
extern int ipx_bind(struct Socket *sk,int recvPacketType,int port,register struct RiscIPX *ri);
extern int ipx_listen(struct Socket *sk,register struct RiscIPX *ri);
extern int ipx_accept(struct Socket *sk,register struct RiscIPX *ri);
extern int ipx_connect(struct Socket *sk,int sendPacketType,const struct Address *name,register struct RiscIPX *ri);
extern int ipx_getport(struct Socket *sk);

/* ----- Send and receive packets ------------------------------------------- */

extern struct mbuf *ipx_recv_mbuf(struct Socket *sk,register struct RiscIPX *ri);
extern int ipx_recv(struct Socket *sk,void *buffer,int length,register struct RiscIPX *ri);

extern int ipx_send_mbuf(struct Socket *sk,struct mbuf *datamb,register struct RiscIPX *ri);
extern int ipx_send(struct Socket *sk,void *data,int length,register struct RiscIPX *ri);
extern int ipx_send_unsafe(struct Socket *sk,void *data,int length,register struct RiscIPX *ri);
extern int ipx_send_pack(struct Socket *sk,void *data,char *format,register struct RiscIPX *ri);
extern int ipx_get_mtu(register struct RiscIPX *ri);


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Routing for IPX.                                                     |
   |                                                                        |
   +------------------------------------------------------------------------+ */

extern int ipxrtr_set_local_net(int frameType,long network,register struct RiscIPX *ri);
extern struct Route *ipxrtr_get_local_net(int frameType,register struct RiscIPX *ri);
extern struct Route *ipxrtr_lookup(long network,register struct RiscIPX *ri);
extern struct Route *ipxrtr_add_route(long network,char *node,register struct RiscIPX *ri);
extern int ipxrtr_delete(long network,register struct RiscIPX *ri);

/* ----- Utility functions -------------------------------------------------- */

extern int memcmp(const void *,const void *,int);

extern int AtoIPX(struct Address *address,const char *string);

extern int Unpack(struct mbuf *source,void *result,const char *format,char *strings);
extern int UnpackN(struct mbuf *source,void *result,const char *format,char *strings,int offset);
extern int UnpackB(void *source,void *result,const char *format,char *strings,int offset);

extern int Pack(struct mbuf *result,const void *data,const char *format);
extern int PackN(struct mbuf *result,const void *data,const char *format,int offset);
extern int PackB(void *result,const void *data,const char *format);

extern int CountPack(const char *format,const void *data);

/* format characters understood by the pack/unpack/count functions:
   'A' : address (32+48+16 bit, long aligned, same as L6bW)
   'b' : byte
   'c' : counted string (with one-byte length preceding the string)
   'e' : align to next address in *data that is a multiple of (count+1)
   'k' : constant (next byte in format string)
   'L' : long (big-endian, 32 bit, long aligned)
   'l' : long (little-endian, 32 bit, long aligned)
   'n' : set count (an int) from the structure, but does not have any packed data
   's' : fixed length string (zero padded to multiplier count, or truncated)
         The structure holds a string pointer; Unpack stores the string in the strings buffer
   'W' : word (big-endian, 16 bit, word aligned)
   'w' : word (little-endian, 16 bit, word aligned)
   'z' : zero byte
   '*' : previous byte, word or long (must be less than 1536) specifies
         multiplier count of next item.  
         BUG: if count is zero, then the next formatting byte is skipped
         (only a problem if you use *w *W *l *L followed by more aligned data)
   '0'..'9' : one or two digits to specify multiplier count of next item
*/

#define __str(x) #x
#define _str(x) __str(x)

/* ----- EOF ----- */
