/* #------------------------------------------------------------------------#
   |                                                                        |
   |   NCP.C                                                                |
   |                                                                        |
   |   Basic NCP functions for NWClient.                                    |
   |                                                                        |
   |   Copyright 1998, Frank A. Vorstenbosch.                               |
   |                                                                        |
   #------------------------------------------------------------------------# */

/* $Id: ncp.c,v 0.41 2001/12/22 12:19:50 frank Exp $ */

#define VERSION 0x00108

#include "nwclient.h"
#include "mbuf.h"

static char OurName[]="NWClient";

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Init and exit functions.                                             |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct NWClient *nwclient_init(register struct RiscIPX *ri)
{  register struct NWClient *nw;
   int buffer;

   Trace("\n nwclient_init() {");

   if((nw=malloc(sizeof(struct NWClient)))==NULL)
      return NULL;

   #ifdef DEBUG
   for(buffer=0;buffer<sizeof(struct NWClient);buffer+=4)
      ((int *)nw)[buffer/4]='CWN!';
   #else
   memset((char *)nw,0,sizeof(struct NWClient));
   #endif

   nw->ri=ri;

   NewList(&nw->Servers);
   NewList(&nw->Mounts);
   nw->Retries=10;   /* 10 retries -> 7.5 second wait before prompting the user */
   nw->Flags=0;
   nw->NumServers=0;
   nw->CurrentMount=NULL;

   nw->PrintDefaults.Banner=NULL;
   nw->PrintDefaults.Form=NULL;
   nw->PrintDefaults.Name=NULL;
   nw->PrintDefaults.Printer=NULL;
   nw->PrintDefaults.Queue="*";
   nw->PrintDefaults.Server=NULL;
   nw->PrintDefaults.Copies=1;
   nw->PrintDefaults.Tabs=0;
   nw->PrintDefaults.FF=0;
   nw->PrintDefaults.Notify=1;
   nw->PrintDefaults.Lines=66;
   nw->PrintDefaults.Columns=80;
   nw->TimeOffset=ReadCurrentTimeZone()/100;

   ncpfs_init(nw);

   AddFilingSystem(nw);
   RegisterFree(nw);

   if(ReadVarVal("NWClient$Stream",&buffer,1<<31,0)==0)
   {  Trace("\n Don't stream");
      nw->Flags|=NWC_DONT_STREAM; }
   else
      Trace("\n Stream");

   SetVarVal("NWClient$Version",(const char *)&Version,4,1);
   ServiceCall(SERVICE_RMLOAD,NWCLIENT_SERVICE_CALL,0);

   Trace(" }");

   return nw;
}

/* -------------------------------------------------------------------------- */

int nwclient_exit(register struct NWClient *nw)
{  char *buffer,*cx;
   struct Server *fs,*temp;

   if(nw->Flags&NWC_SET_CALLBACK)
   {  nw->Flags&=~NWC_SET_CALLBACK;
      RemoveCallBack(nw); }

   ncpfs_exit(nw);

   DeregisterFree(nw);
   RemoveFilingSystem(nw);

   forList2(&nw->Servers,fs,temp)
   {  nwclient_logout(fs,nw);
      nwclient_detach(fs,nw); }

   ServiceCall(SERVICE_RMKILL,NWCLIENT_SERVICE_CALL,0);

   /* now we have all of the nw structure available for temporary storage
      and we'll use it to clear out any unwanted environment variables */

   cx=NULL;
   buffer=(char *)nw;

   while(1)
   {
      /*ReadVarVal("*",buffer,sizeof(struct NWClient),cx,0)==0)*/
      break;
   }

   free(nw);
   return 0;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   NCP packet dump for debugging.                                       |
   |                                                                        |
   +------------------------------------------------------------------------+ */

#if defined DEBUG && ((DEBUG&16) || (DEBUG&4))
static void DumpByte(int b)
{  TraceChar(((b>>4)&15)>9?((b>>4)&15)+'a'-10:((b>>4)&15)+'0');
   TraceChar((b&15)>9?(b&15)+'a'-10:(b&15)+'0'); }
#endif

#if defined DEBUG && (DEBUG&16)
static void DumpShort(int b)
{  DumpByte(b>>8);
   DumpByte(b);
}

static void DumpLong(int b)
{  DumpShort(b>>16);
   DumpShort(b);
}

#if 0
static void DumpInt(int i)
{  int c,n,f=0;

   if(i<0 || i>=10000)
      i=9999;

   for(n=1000;n>1;)
   {  for(c='0';i>=n;c++)
         i-=n;
      if(f || c>'0')
      {  f=1;
         TraceChar(c); }
      switch(n)
      {  case 1000:
            n=100;
            break;
         case 100:
            n=10;
            break;
         case 10:
            n=1;
            break;
      }
   }
   TraceChar(i+'0');
}
#endif

static void DumpMAC(char *mac)
{  DumpByte(*mac++);
   TraceChar(':');
   DumpByte(*mac++);
   TraceChar(':');
   DumpByte(*mac++);
   TraceChar(':');
   DumpByte(*mac++);
   TraceChar(':');
   DumpByte(*mac++);
   TraceChar(':');
   DumpByte(*mac);
}

static void DumpIPXaddr(char *addr)
{
   DumpByte(addr[0]);
   DumpByte(addr[1]);
   DumpByte(addr[2]);
   DumpByte(addr[3]);
   TraceChar(':');
   DumpMAC(addr+4);
   TraceChar(':');
   DumpByte(addr[10]);
   DumpByte(addr[11]);
}

void DumpRequest(struct NWClient *nw,char *requestFormat,char *data)
{  char *p,*q;
   int i,count=1,close=0,prev=1,save;

   p=data;
   q=requestFormat;
   TraceChar('"');
   Trace(q);
   Trace("\"\n");

   while(*q)
   {  if(count<1)
      {  Trace("() ");
         q++;
         continue; }
      if(count>512)
         count=512;
      switch(*q)
      {  case 'A': /* address (32+48+16 bit, long aligned, same as L6bW) */
            p=(char *)((int)(p+3)&~3);
            if(count>1)
            {  close=1;
               Trace("("); }
            while(count--)
            {  DumpIPXaddr(p);
               if(count)
                  TraceChar(' ');
               p+=12; }
            if(close)
            {  close=0;
               Trace(") "); }
            else
               TraceChar(' ');
            count=1;
            break;
         case 'b': /* byte */
            if(count>1)
            {  close=1;
               Trace("("); }
            while(count--)
            {  DumpByte(prev=*p++);
               if(count)
                  TraceChar(' '); }
            if(close)
            {  close=0;
               Trace(") "); }
            else
               TraceChar(' ');
            count=1;
            break;
         case 'c': /* counted string (with one-byte length preceding the string) */
         case 's': /* fixed length string (zero padded to multiplier count, or truncated) */
            p=(char *)((int)(p+3)&~3);
            Trace(" \"");
            Trace(*((char **)p)++);
            Trace("\" ");
            count=1;
            break;
         case 'e': /* align to next address in *data that is a multiple of (count+1) */
            p=(char *)((int)(p+count)&~count);
            count=1;
            break;
         case 'k': /* constant (next byte in format string) */
            i=q[1];
            q++;
            if(count>1)
            {  close=1;
               Trace("("); }
            while(count--)
            {  DumpByte(i);
               if(count)
                  TraceChar(' '); }
            if(close)
            {  close=0;
               Trace(") "); }
            else
               TraceChar(' ');
            count=1;
            break;
         case 'L': /* long (big-endian, 32 bit, long aligned) */
         case 'l': /* long (little-endian, 32 bit, long aligned) */
            p=(char *)((int)(p+3)&~3);
            if(count>1)
            {  close=1;
               Trace("("); }
            while(count--)
            {  DumpLong(prev=*((int *)p)++);
               if(count)
                  TraceChar(' '); }
            if(close)
            {  close=0;
               Trace(") "); }
            else
               TraceChar(' ');
            count=1;
            break;
         case 'W': /* word (big-endian, 16 bit, word aligned)    */
         case 'w': /* word (little-endian, 16 bit, word aligned) */
            p=(char *)((int)(p+1)&~1);
            if(count>1)
            {  close=1;
               Trace("("); }
            while(count--)
            {  DumpShort(prev=*((short *)p)++);
               if(count)
                  TraceChar(' '); }
            if(close)
            {  close=0;
               Trace(") "); }
            else
               TraceChar(' ');
            count=1;
            break;
         case 'z': /* zero byte */
            if(count>1)
            {  close=1;
               Trace("("); }
            while(count--)
            {  DumpByte(0);
               if(count)
                  TraceChar(' '); }
            if(close)
            {  close=0;
               Trace(") "); }
            else
               TraceChar(' ');
            count=1;
            break;
         case '*': /* previous byte, word or long (must be less than 1536) specifies */
            count=prev;
            break;
         case '0'..'9': /* one or two digits to specify multiplier count of next item */
            count=*q-'0';
            if(q[1]>='0' && q[1]<='9')
            {  q++;
               count=(count<<3)+(count<<1)+*q-'0'; }
            break;
         case '=':
            save=count;
            break;
         case '?':
            count=save;
            break;
         case 'n':
            p=(char *)((int)(p+3)&~3);
            prev=*((int *)p)++;
            Trace(" [");
            TraceVal(prev);
            Trace("] ");
            count=1;
            break;
         default:
            Trace("??? ");
            Break();
      }
      q++;
   }

   TraceChar('^');
}
#endif

#if defined DEBUG && ((DEBUG&16) || (DEBUG&4))
static void DumpLine(char *bytes,int offset,int len)
{  int i;

   TraceChar('\n');
   TraceChar((offset>>8)+'0');
   DumpByte(offset);
   Trace(": ");

   for(i=0;i<len;i++)
   {  DumpByte(bytes[i]);
      Trace(i==7?"  ":" "); }
   for(;i<16;i++)
      Trace(i==7?"    ":"   ");
   Trace("  ");
   for(i=0;i<len;i++)
      TraceChar(bytes[i]<32?'.':bytes[i]);
}
#endif


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Main function to process NCP requests.                               |
   |                                                                        |
   +------------------------------------------------------------------------+ */

/* returns negative value on error, or 0 if all OK */

int _ncp_request(char *requestFormat,char *replyFormat,char *strings,register struct Server *fs)
{  struct NWClient *nw;
   struct RiscIPX *ri;
   struct mbuf *mb=NULL;
   int i,tries,wait,errorbox=0;

   Trace("\n NCP Request ");

   if(!fs || !fs->NcpSocket)
      return IPX_NO_SOCKET;
   
   #if defined DEBUG && DEBUG&4
   DumpByte(fs->request.header.Sequence);
   #endif

   nw=fs->nw;

   if(fs->Flags&SERVER_TCP_IP)
   {
      UNIM: TCP/IP
   }
   else
   {
      ri=nw->ri;

      do
      {
         for(tries=0;!mb && tries<nw->Retries;tries++)
         {
            #if defined DEBUG && (DEBUG&16)
            Trace(": ");
            DumpRequest(nw,requestFormat,(char *)&fs->request);
            #endif

            if((i=ipx_send_pack(fs->NcpSocket,&fs->request,requestFormat,ri))<0)
            {  
               if(errorbox)
                  ReportError(&Error_WaitingNCPReq,0xd3,OurName);
               #if defined DEBUG && (DEBUG&4)
               ShowError("ncp_req_send",i);
               #endif
               return i; }

            for(wait=0;!mb && wait<4*fs->Timeout;wait++)
            {  
               int now=MonotonicTime();

               while(now==MonotonicTime() && (mb=ipx_recv_mbuf(fs->NcpSocket,ri))==NULL)
                  ;

               if(mb)
               {
                  if((i=UnpackN(mb->next,&nw->reply,"W6b",NULL,mb->pkthdr.len))>0)
                  {
                     #if 0
                     Trace(" reqseq=");
                     TraceByte(fs->request.header.Sequence);
                     Trace(" rplseq=");
                     TraceByte(nw->reply.header.Sequence);
                     #endif

                     if(fs->request.header.Sequence==nw->reply.header.Sequence)
                     {
                        if(nw->reply.header.Type==NCP_REPLY)
                        {  fs->request.header.Sequence++;

                           /* now we adjust the timeout down, but only if this was the first try */

                           #define DOWN_ADJUST_TIMECONST 4

                           if(!tries)
                           {  if((nw->reply.header.Sequence&((1<<DOWN_ADJUST_TIMECONST)-1))==0)
                              {
                                 Trace("wait=");TraceVal(wait);

                                 if(wait>1 && wait<4*fs->Timeout && fs->Timeout>250/40)
                                 {
                                    #ifdef DEBUG
                                    Trace(" timeout--");
                                    #endif
                                    fs->Timeout--;
                                 }
                              }
                           }

                           if(replyFormat && nw->reply.header.CompletionCode==0)
                           {  
                              nw->ReplyLength=i=UnpackN(mb->next,((char *)&nw->reply)+8,replyFormat,strings,mb->pkthdr.len+8);
                              #if defined DEBUG && DEBUG&4
                              if(i<0)
                              {  ShowError("error=",i);
                                 nw->ReplyLength=0; }
                              #endif
                              #if defined DEBUG && DEBUG&16
                              Trace("\n NCP Reply: ");
                              if(i>=0)
                                 DumpRequest(nw,replyFormat,((char *)(&nw->reply))+8);
                              #endif
                           }
                           else
                              nw->ReplyLength=i=0;
                           ri->mbctl.freem(&ri->mbctl,mb);

                           if(errorbox)
                              ReportError(&Error_WaitingNCPReq,0xd3,OurName); 

                           if(nw->reply.header.ConnectionState&CONSTATE_MESSAGE &&
                              !(nw->Flags&NWC_GET_MESSAGE))
                           {  nw->Flags|=NWC_GET_MESSAGE;
                              AddCallBack(nw); }

                           #if defined DEBUG && (DEBUG&4)
                           if(nw->reply.header.CompletionCode)
                              ShowError("ncp_reply",-nw->reply.header.CompletionCode);
                           #endif

                           return nw->reply.header.CompletionCode?-nw->reply.header.CompletionCode:(i<0?i:0); }
                        }
                        else if(nw->reply.header.Type==NCP_POSITIVE_ACK)
                           tries=1;
                     }

                  ri->mbctl.freem(&ri->mbctl,mb);
                  mb=NULL;
               }

               if(errorbox)
               {  
                  switch(ReportError(&Error_WaitingNCPReq,0xb3,OurName))
                  {  case 2:
                        ReportError(&Error_WaitingNCPReq,0xd3,OurName);
                        return NCP_TIMEOUT_FAILURE; 
                     case 1:
                        ReportError(&Error_WaitingNCPReq,0xd3,OurName);
                        wait=tries=errorbox=0;
                  }
               }
            }

            /* when we get here, the initial timeout has passed, and we
               re-send the packet.  we also adjust the timeout count,
               but only on the first retry. */

            if(!(nw->Flags&NWC_SET_CALLBACK))
               ForcePoll();   /* make sure callback handlers get executed */

            #if 0
            if(!tries && (3*fs->Timeout)/2<256)
               fs->Timeout=(3*fs->Timeout)/2;
            #else
            if(fs->Timeout<255)
            {
               Trace(" wait=");TraceVal(wait);

               #ifdef DEBUG
               Trace(" timeout++");
               #endif
               fs->Timeout++;
            }
            #endif
         }

      } while(errorbox=1,ReportError(&Error_WaitingNCPReq,0xb3,OurName)!=2);
   }
   
   /* we drop out here if cancel is selected on the error box */

   ReportError(&Error_WaitingNCPReq,0xd3,OurName);
   return NCP_TIMEOUT_FAILURE;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Receive watchdog packet.                                             |
   |                                                                        |
   +------------------------------------------------------------------------+ */

#define WATCHDOG_POLL            '?'
#define WATCHDOG_GET_MESSAGE     '!'
#define WATCHDOG_SESSION_VALID   'Y'
#define WATCHDOG_NO_SESSION      'N'

int wdog_handler(struct mbuf *mb,struct UnpackedHeader *header,void *fsptr)
{  struct Server *fs=fsptr;
   struct NWClient *nw=fs->nw;
   char WdogPayload[2];
   int i;

   #if defined DEBUG
   Trace("\n wdog");
   if((i=UnpackN(mb->next,WdogPayload,"2b",NULL,mb->pkthdr.len))>0)
   {  TraceByte(WdogPayload[0]);
      TraceByte(WdogPayload[1]); }
   #endif

   /* Note: How about 2-byte connection numbers? */
   if((i=UnpackN(mb->next,WdogPayload,"2b",NULL,mb->pkthdr.len))>0 &&
      WdogPayload[0]==fs->ConnLow && 
      (WdogPayload[1]==WATCHDOG_POLL || WdogPayload[1]==WATCHDOG_GET_MESSAGE))
   {  
      /* now we need to send a watchdog reply, but better not inside the
         interrupt handler... */
      #if defined DEBUG && (DEBUG&2)
      Trace(" adding");
      #endif

      if(!(fs->Flags&SERVER_BIND_WDOG))
      {  fs->Flags|=SERVER_BIND_WDOG;
         #if defined DEBUG
         Trace("\nBind wdog socket");
         #endif         
         ipx_connect(fs->WdogSocket,IPX_PT_PEP,&header->Source,nw->ri);
         ipx_connect(fs->MsgSocket,IPX_PT_PEP,&header->Source,nw->ri);
      }

      if(WdogPayload[1]==WATCHDOG_GET_MESSAGE)
         fs->Flags|=SERVER_GET_MESSAGE;

      if(!(nw->Flags&NWC_SET_CALLBACK))
      {  AddCallBack(nw);
         nw->Flags|=NWC_SET_CALLBACK; }
   }

   #if defined DEBUG && (DEBUG&2)
   Trace(" freeing");
   #endif
   nw->ri->mbctl.freem(&nw->ri->mbctl,mb);

   return 0;
}

/* -------------------------------------------------------------------------- */

void wdog_callback(register struct NWClient *nw)
{  char WdogPayload[2];
   struct Server *fs;

   #if defined DEBUG
   Trace("\n wdog_callback");
   #endif

   forList(&nw->Servers,fs)
   {  if(fs->Flags&SERVER_GET_MESSAGE)
      {  while(fs->Flags&SERVER_GET_MESSAGE)
         {
            #if defined DEBUG
            Trace("\n wdog_get_message");
            #endif

            fs->request.header.Function=0x15;
            fs->request.payload.fn_15.Length=1;
            fs->request.payload.fn_15.SubFunction=0x0b;  /* get broadcast message */

            if(ncp_request("Wb","c",&nw->reply.payload.Data[4],fs)>=0)
            {
               #if defined DEBUG && (DEBUG&2)
               Trace("\ngot the message ");
               Trace(&nw->reply.payload.Data[4]);
               #endif
               fs->Flags&=~SERVER_GET_MESSAGE;

               nwclient_logmessage(&nw->reply.payload.Data[4]);

               if(nw->reply.header.ConnectionState&CONSTATE_DISCONNECTED)
               {  
                  nwclient_detach(fs,nw);

                  WdogPayload[0]=fs->ConnLow;
                  WdogPayload[1]=WATCHDOG_NO_SESSION;
                  ipx_send_pack(fs->WdogSocket,&WdogPayload,"bb",nw->ri);

                  #if 0
                  ReportError((struct kernel_error *)&nw->reply.payload.Data[0],0x19,OurName);
                  #endif
               }

               {  int time=MonotonicTime()+60*100;  /* messages only get 60 seconds of fame */

                  while(MonotonicTime()<time &&
                        ReportError((struct kernel_error *)&nw->reply.payload.Data[0],0x39,OurName)==0)
                  {  /* nop */ }

                  ReportError((struct kernel_error *)&nw->reply.payload.Data[0],0x79,OurName);
               }
            }
            else
               fs->Flags&=~SERVER_GET_MESSAGE;
         }
      }
      else
      {
         #if defined DEBUG
         Trace("\n wdog_reply");
         #endif
   
         WdogPayload[0]=fs->ConnLow;
         WdogPayload[1]=WATCHDOG_SESSION_VALID;
         ipx_send_pack(fs->WdogSocket,&WdogPayload,"bb",nw->ri);
      }
   }

   #if defined DEBUG && (DEBUG&2)
   Trace(" --OK");
   #endif

   nw->Flags&=~NWC_SET_CALLBACK;
}


/* -------------------------------------------------------------------------- */

int make_path(char *nwpath,const char *name)
{  char *p,*size;
   const char *q;

   *nwpath=0;
   size=nwpath+1;
   p=nwpath+2;
   q=name;

   while(*q)
   {  (*nwpath)++;
      *size=0;
      while(*q && *q!=':' && *q!='/' && *q!='\\')
      {  *p++=*q++;
         (*size)++; }
      while(*q==':' || *q=='/' || *q=='\\')
         q++;
      size=p++;
   }

   return size-nwpath;
}

/* -------------------------------------------------------------------------- */

int ncp_connection_info(struct user_lookup **user,int connection,struct Server *fs,register struct NWClient *nw)
{  int r;

   Trace("\n ncp_connection_info(fs=");
   TraceVal((int)fs);
   Trace(" nw=");
   TraceVal((int)nw);
   Trace(") {");

   fs->request.header.Function=0x17;
   fs->request.payload.fn_17.Length=5;
   fs->request.payload.fn_17.SubFunction=0x1c;     /* get connection info */
   fs->request.payload.fn_17_1c.Connection=connection;

   if((r=ncp_request(REQ_17"l","lw48b7b",NULL,fs))==NCP_NCP_NOT_SUPPORTED)
   {  /* this will happen on 3.x servers */
      fs->request.header.Function=0x17;
      fs->request.payload.fn_17.Length=2;
      fs->request.payload.fn_17.SubFunction=0x16;     /* get connection info (old) */
      fs->request.payload.fn_17_16.Connection=connection;
      r=ncp_request(REQ_17"b","lw48b7b",NULL,fs);
      /* fortunately the reply is the same for both 17/16 and 17/1c */
   }

   *user=&nw->reply.payload.fn_17_1c;
   return r;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Directory scanning.                                                  |
   |                                                                        |
   +------------------------------------------------------------------------+ */

int nwclient_init_search(struct Mount *mt,const char *directory,char Sequence[10],struct NWClient *nw)
{  int r;
   struct Server *fs=mt->Server;

   #if defined DEBUG && (DEBUG&2)
   Trace("\n nwclient_init_search(");
   Trace(directory);
   Trace(") ");
   #endif

   fs->request.header.Function=0x57;
   fs->request.payload.fn_57.SubFunction=2;   /* init search */

   if(mt)
   {  Sequence[9]=fs->request.payload.fn_57_02.NameSpace=(mt->Flags&MOUNT_LONGNAMES);
      fs->request.payload.fn_57_02.Volume=mt->Volume;
      fs->request.payload.fn_57_02.Directory=mt->Directory;
      fs->request.payload.fn_57_02.HandleFlag=1; }
   else
   {  Sequence[9]=fs->request.payload.fn_57_02.NameSpace=0;
      fs->request.payload.fn_57_02.Volume=-1;
      fs->request.payload.fn_57_02.Directory=0;
      fs->request.payload.fn_57_02.HandleFlag=-1; }
   fs->request.payload.fn_57_02.PathBytes=make_path(&fs->request.payload.fn_57_02.PathComponents,directory);

   if((r=ncp_request(REQ_57"bzblbn*b","9b",NULL,fs))<0)
   {  
      #if defined DEBUG && (DEBUG&4)
      ShowError("nwclient_init_search",r);
      #endif
      return r; }

   memcpy(Sequence,nw->reply.payload.fn_57_02.Sequence,9);

   return 0;
}

int nwclient_init_search_num(struct Mount *mt,int directory,char Sequence[10],struct NWClient *nw)
{  int r;
   struct Server *fs=mt->Server;

   #if defined DEBUG && (DEBUG&2)
   Trace("\n nwclient_init_search(");
   TraceVal(directory);
   Trace(") ");
   #endif

   fs->request.header.Function=0x57;
   fs->request.payload.fn_57.SubFunction=2;   /* init search */

   Sequence[9]=fs->request.payload.fn_57_02.NameSpace=(mt->Flags&MOUNT_LONGNAMES);
   fs->request.payload.fn_57_02.Volume=mt->Volume;
   fs->request.payload.fn_57_02.Directory=directory;
   fs->request.payload.fn_57_02.HandleFlag=1;
   fs->request.payload.fn_57_02.PathBytes=fs->request.payload.fn_57_02.PathComponents=0;

   if((r=ncp_request(REQ_57"bzblbn*b","9b",NULL,fs))<0)
   {  
      #if defined DEBUG && (DEBUG&4)
      ShowError("nwclient_init_search",r);
      #endif
      return r; }

   memcpy(Sequence,nw->reply.payload.fn_57_02.Sequence,9);

   return 0;
}

/* -------------------------------------------------------------------------- */

int nwclient_search(struct Mount *mt,char *name)
{  int r;
   struct Server *fs=mt->Server;

   #if defined DEBUG && (DEBUG&2)
   Trace("\n nwclient_search(");
   TraceVal((int)name);
   Trace(") ");
   #endif

   #undef ENTRY_INFO
   #undef RIM_ALL
   #define RIM_ALL      (RIM_NAME|RIM_ATTRIBUTES)
   #define ENTRY_INFO "77zc"

   fs->request.header.Function=0x57;
   fs->request.payload.fn_57.SubFunction=3;   /* search */
   fs->request.payload.fn_57_03.NameSpace=mt->Sequence[9];
   fs->request.payload.fn_57_03.SearchAttributes=0x8006;
   fs->request.payload.fn_57_03.RIM=RIM_ALL;
   memcpy(fs->request.payload.fn_57_03.Sequence,mt->Sequence,9);

   if((r=ncp_request(REQ_57"bzwl9bk\002k\377k*","9b"ENTRY_INFO,name,fs))<0)
   {  
      #if defined DEBUG && (DEBUG&4)
      ShowError("nwclient_search",r);
      #endif
      return r; }

   #if defined DEBUG && DEBUG&2
   Trace(" name->\"");
   Trace(name);
   Trace("\"");
   #endif

   memcpy(mt->Sequence,fs->nw->reply.payload.fn_57_03.Sequence,9);
	return 0;
}

/* ----- EOF NCP.C ----- */
