/* #------------------------------------------------------------------------#
   |                                                                        |
   |   FSMAIN.C                                                             |
   |                                                                        |
   |   Functions to translate RiscOS calls to NCP ones.                     |
   |                                                                        |
   |   Copyright 2001, Frank A. Vorstenbosch                                |
   |                                                                        |
   #------------------------------------------------------------------------# */

/* $Id: fsmain.c,v 1.20 2001/12/24 16:52:38 frank Exp $ */

#define VERSION 0x00090

#include "nwclient.h"

/* +---------------------------------------------------------------------+
   |                                                                     |
   |   Initialize.                                                       |
   |                                                                     |
   +---------------------------------------------------------------------+ */

void ncpfs_init(register struct NWClient *nw)
{
   Trace("\n ncpfs_init() {");
   DefaultExtensions(nw);
   Trace(" }");
}

void ncpfs_exit(struct NWClient *nw)
{
   if(nw->ExtensionMap)
      free(nw->ExtensionMap);
}

/* +---------------------------------------------------------------------+
   |                                                                     |
   |   Fileswitch functions.                                             |
   |                                                                     |
   +---------------------------------------------------------------------+ */

#pragma On(Check_ltorg)                  /* most of these functions are too big */

struct kernel_error *ReadCatInfo(register struct NWClient *nw,const char *fileName)
{  char buffer[FILENAME_MAX];
   struct Mount *mt;
   struct Server *fs;
   struct File *file;
   int date[2];

   #if defined DEBUG && DEBUG&2
   Trace("\n ReadCatInfo(\"");
   Trace(fileName);
   Trace("\") ");
   #endif

   if((mt=RosToNWName(buffer,fileName,nw,0))==NULL)
   {  nw->ReturnStruct[0]=0;
      return &Error_InvalidName; }

   fs=mt->Server;

again:
   if(!*buffer)
   {  nw->ReturnStruct[0]=2;
      nw->ReturnStruct[2]=nw->ReturnStruct[3]=nw->ReturnStruct[4]=nw->ReturnStruct[5]=0;
      return NULL; }

   fs->request.header.Function=0x57;
   fs->request.payload.fn_57.SubFunction=0x06;     /* get info */
   fs->request.payload.fn_57_06.NameSpace=         /* namespace (0 for DOS or 4 for OS/2 */
      fs->request.payload.fn_57_06.DestNameSpace=(mt->Flags&MOUNT_LONGNAMES); 
   fs->request.payload.fn_57_06.SearchAttributes=0x8006;
   fs->request.payload.fn_57_06.RIM=RIM_ALL;
   fs->request.payload.fn_57_06.Volume=mt->Volume;
   fs->request.payload.fn_57_06.Directory=mt->Directory;
   fs->request.payload.fn_57_06.HandleFlag=1;      /* have a dirbase */
   fs->request.payload.fn_57_06.PathBytes=make_path(&fs->request.payload.fn_57_06.PathComponents,buffer);
   if(ncp_request("3bwl"REQ_PATH,ENTRY_INFO_NO_NAME,NULL,fs)) 
   {  nw->ReturnStruct[0]=0;
      return NULL; }

   nw->ReturnStruct[0]=(nw->reply.payload.fn_57_06.Attributes&ATTR_DIR)?2:1;

   NWToRosDate(date,nw->reply.payload.fn_57_06.ModifyDate,nw->reply.payload.fn_57_06.ModifyTime,nw->TimeOffset);

   if(nw->ReturnStruct[0]==1)
   {  
      forList(&mt->Files,file)
      {  
         if(file->DirEntNum==nw->reply.payload.fn_57_06.DirEntNum &&
            file->VolNumber==nw->reply.payload.fn_57_06.VolNumber)
         {  nw->reply.payload.fn_57_06.DataStreamSize=file->CurrentSize;
            break; }
   }  }

   nw->ReturnStruct[2]=0xfff00000|date[1];
   nw->ReturnStruct[3]=date[0];

   nw->ReturnStruct[4]=nw->reply.payload.fn_57_06.DataStreamSize;
   nw->ReturnStruct[5]=RosAttrib(nw->reply.payload.fn_57_06.Attributes);

   /* for internal use only, not returned to RiscOS */
   nw->ReturnStruct[6]=nw->reply.payload.fn_57_06.DirEntNum;
   nw->ReturnStruct[7]=nw->reply.payload.fn_57_06.VolNumber;

   if(!(nw->reply.payload.fn_57_06.Attributes&ATTR_DIR))
   {  
      #if defined DEBUG && DEBUG&2
      Trace("\n Attributes");
      TraceVal(nw->reply.payload.fn_57_06.Attributes);
      if(nw->reply.payload.fn_57_06.Attributes&(ATTR_SHARED|ATTR_HIDDEN|ATTR_SYSTEM))
         Trace(" Special");
      #endif

      switch(nw->reply.payload.fn_57_06.Attributes&(ATTR_SHARED|ATTR_HIDDEN|ATTR_SYSTEM))
      {  case ATTR_SHARED:
            /* just informative */
            break;
         case ATTR_SHARED|ATTR_HIDDEN:
            if(nw->reply.payload.fn_57_06.DataStreamSize>8 &&
               nw->reply.payload.fn_57_06.DataStreamSize<=NCP_MAX_SYMLINK_SIZE)
            #if 0
            {  struct File *file;
               int i;
               char *p,*q;

               if((file=nwclient_open(mt,buffer,O_OPEN|O_READ,nw))!=NULL)
               {  
                  i=nwclient_read(file,nw->SmallBuffer,0,NCP_MAX_SYMLINK_SIZE,nw);
                  nwclient_close(file,nw);

                  if(i>8 && ((int *)nw->SmallBuffer)[0]==SYMLNK_MAGIC0 &&
                            ((int *)nw->SmallBuffer)[1]==SYMLNK_MAGIC1)
                  {  nw->SmallBuffer[i]=0;
                     Trace("\n Symbolic link = \"");
                     Trace(nw->SmallBuffer);
                     Trace("\" ");

                     strcpy(fs->request.payload.Small.Data,nw->SmallBuffer+8);
                     strcpy(nw->SmallBuffer,buffer);
                     p=nw->SmallBuffer;
                     i=0;
                     while(*p)
                     {  if(*p=='/')
                           i=p-nw->SmallBuffer;
                        p++; }
                     nw->SmallBuffer[i+1]=0;

                     q=fs->request.payload.Small.Data;
                     while(q[0]=='.' && q[1]=='.' && q[2]=='/')
                     {  q+=3;
                        i=0;
                        while(*p)
                        {  if(*p=='/')
                              i=p-nw->SmallBuffer;
                           p++; }
                        nw->SmallBuffer[i+1]=0;
                     }

                     strcpy(nw->SmallBuffer+i+1,fs->request.payload.Small.Data);
                     nw->SmallBuffer[FILENAME_MAX-1]=0;
                     strcpy(buffer,nw->SmallBuffer);
                     Trace("\n New path:\"");
                     Trace(buffer);
                     Trace("\" ");

                     goto again;
                  }
               }

               nw->ReturnStruct[2]|=SOFTLINK<<8;  /* softlink */
            }
            #else
               nw->ReturnStruct[2]|=SOFTLINK<<8;  /* softlink */
            #endif
            break;
         case ATTR_SHARED|ATTR_SYSTEM:
            nw->ReturnStruct[2]|=UNIXEXEC<<8;  /* unix exec */
            break;
         case ATTR_SHARED|ATTR_HIDDEN|ATTR_SYSTEM:
            /* reserved case */
            break;
      }

      /* buffer[] still holds the NetWare name */

      if(!(nw->ReturnStruct[2]&0x000fff00))
      {
         fs->request.header.Function=0x56;
         fs->request.payload.fn_56_03.SubFunction=0x03;
         fs->request.payload.fn_56_03.Flags=0;
         fs->request.payload.fn_56_03.Is.Entry.Volume=nw->reply.payload.fn_57_06.VolNumber;
         fs->request.payload.fn_56_03.Is.Entry.DirEntry=nw->reply.payload.fn_57_06.DosDirNum;
         fs->request.payload.fn_56_03.Position=0;
         fs->request.payload.fn_56_03.Length=8;
         if((mt->Flags&MOUNT_LONGNAMES) && ncp_request("bw4lk\006zkrkikskckoks","4lw=3e?*b",NULL,fs)>=0)
         {  fs->request.header.Function=0x56;
            fs->request.payload.fn_56_02.SubFunction=1;
            fs->request.payload.fn_56_01.EAHandle=nw->reply.payload.fn_56_03.EAHandle;
            ncp_request("bzzl",NULL,NULL,fs);

            #if defined DEBUG && DEBUG&2
            if(nw->reply.payload.fn_56_03.Length==8)
            {  Trace("\n Load/execute=");
               TraceVal(nw->reply.payload.fn_56_03.Contents.FullType[0]);
               TraceVal(nw->reply.payload.fn_56_03.Contents.FullType[1]); }
            else if(nw->reply.payload.fn_56_03.Length==2)
            {  
               Trace("\n Test is");
               TraceVal(nw->reply.payload.fn_56_03.Contents.Test[0]);
               TraceVal(nw->reply.payload.fn_56_03.Contents.Test[1]);
               TraceVal(nw->reply.payload.fn_56_03.Contents.Test[2]);
               TraceVal(nw->reply.payload.fn_56_03.Contents.Test[3]);
               
               Trace(" pointers");
               TraceVal((int)(&nw->reply.payload.fn_56_03.Contents.Test[0]));
               TraceVal((int)(&nw->reply.payload.fn_56_03.Contents.Value));

               Trace("\n Type is");
               TraceVal(nw->reply.payload.fn_56_03.Contents.Value);}
            else
               Trace("\n Untyped");
            #endif
      
            if(nw->reply.payload.fn_56_03.Length==8)
            {  nw->ReturnStruct[2]=nw->reply.payload.fn_56_03.Contents.FullType[0];
               nw->ReturnStruct[3]=nw->reply.payload.fn_56_03.Contents.FullType[1];
            }
            else if(nw->reply.payload.fn_56_03.Length==2)
            {  if(nw->reply.payload.fn_56_03.Contents.Value&0xf000)
                  nw->ReturnStruct[2]=nw->ReturnStruct[3]=(nw->reply.payload.fn_56_03.Contents.Value==-1)?0:0xdeaddead;
               else
                  nw->ReturnStruct[2]=(nw->ReturnStruct[2]&0xfff000ff)|(nw->reply.payload.fn_56_03.Contents.Value<<8);
            }
            else
               nw->ReturnStruct[2]|=InventFileType(nw,buffer);
         }
         else
            nw->ReturnStruct[2]|=InventFileType(nw,buffer);
      }

   } /* if a file */

   #if defined DEBUG && DEBUG&2
   Trace("\n type,load/exec,len,attrib=");
   TraceVal(nw->ReturnStruct[0]);
   TraceVal(nw->ReturnStruct[2]);
   TraceVal(nw->ReturnStruct[3]);
   TraceVal(nw->ReturnStruct[4]);
   TraceVal(nw->ReturnStruct[5]);
   #endif

   return NULL;
}


/* ----------------------------------------------------------------------- */

struct kernel_error *OpenFile(int fnCode,const char *fileName,struct NWClient *nw,int rosHandle)
{  struct File *file;
   struct Mount *mt;
   struct Server *fs;
   char buffer[FILENAME_MAX];
   int date[2];

   #if defined DEBUG
   Trace("\n OpenFile(");
   Trace(fileName);
   TraceByte(fnCode);
   Trace(") ");
   #endif

   if((mt=RosToNWName(buffer,fileName,nw,0))==NULL)
      return &Error_InvalidName;
   fs=mt->Server;

   if(ReadCatInfo(nw,fileName)!=NULL)
      date[0]=date[1]=0;
   else
   {  date[0]=nw->ReturnStruct[2];
      date[1]=nw->ReturnStruct[3]; }

   #if 0
   fs->request.header.Function=0x57;
   fs->request.payload.fn_57.SubFunction=0x06;     /* get info */
   fs->request.payload.fn_57_06.NameSpace=         /* namespace (0 for DOS or 4 for OS/2 */
      fs->request.payload.fn_57_06.DestNameSpace=(mt->Flags&MOUNT_LONGNAMES); 
   fs->request.payload.fn_57_06.SearchAttributes=0x8006;
   fs->request.payload.fn_57_06.RIM=RIM_MODIFY;
   fs->request.payload.fn_57_06.Volume=mt->Volume;
   fs->request.payload.fn_57_06.Directory=mt->Directory;
   fs->request.payload.fn_57_06.HandleFlag=1;      /* have a dirbase */
   fs->request.payload.fn_57_06.PathBytes=make_path(&fs->request.payload.fn_57_06.PathComponents,buffer);
   if(ncp_request("3bwl"REQ_PATH,ENTRY_INFO,buffer,fs)>=0)
      NWToRosDate(date,nw->reply.payload.fn_57_06.ModifyDate,nw->reply.payload.fn_57_06.ModifyTime,nw->TimeOffset);
   else
      date[0]=date[1]=0;
   #endif

   if((mt=RosToNWName(buffer,fileName,nw,0))==NULL)
      return &Error_InvalidName;

   if((file=nwclient_open(mt,buffer,O_OPEN|(fnCode?O_RDWR:O_READ),nw))==NULL)
   {  nw->ReturnStruct[0]=nw->ReturnStruct[1]=0;

      if(-nw->reply.header.CompletionCode==NCP_FILE_NAME_INVALID)
         return &Error_InvalidName;

      return NULL; }

   if(date[0] || date[1])
   {  file->Flags|=FILE_GOTDATE;
      file->Is.File.DateHi=date[0];
      file->Is.File.DateLo=date[1];
   }
   
   file->DirEntNum=nw->ReturnStruct[6];
   file->VolNumber=nw->ReturnStruct[7];

   Trace(" Openfile, set DirEntNum to");
   TraceVal(file->DirEntNum);

   file->RosHandle=rosHandle;

   nw->ReturnStruct[0]=(file->Flags&FILE_WRITABLE)?3<<30:1<<30;
   nw->ReturnStruct[1]=(int)file;
   nw->ReturnStruct[2]=fs->BlockSize;
   nw->ReturnStruct[3]=file->CurrentSize=nwclient_filesize(file,nw);
   nw->ReturnStruct[4]=(nw->ReturnStruct[3]+nw->ReturnStruct[2]-1)&~(nw->ReturnStruct[2]-1);

   Trace(" checking forList walk");
   forList(&mt->Files,file)
   {  
      Trace(" filename ");
      Trace(file->Name);
      TraceVal(file->DirEntNum);
   }

   return NULL;
}

/* ----------------------------------------------------------------------- */

#define SPOOLCMD 256

struct kernel_error *SpoolFile(int fnCode,const char *fileName,struct NWClient *nw,int rosHandle)
{  struct File *file;
   char *p,command[SPOOLCMD];
   struct kernel_error *err;
   struct NWPrintSettings Settings;
   int i;
   struct
   {  char *banner,
           *form,
           *name,
           *queue;
   #ifdef NDS
      char *printer,
           *server;
   #endif
      struct expression
      {  char zero;
         char value;
      } *copies,
        *tabs,
        *lines,
        *columns;
      int ff,
          nobanner,
          noff,
          nonotify,
          notify;
      char buffer[SPOOLCMD]; 
   } result;

   #if defined DEBUG
   Trace("\n SpoolFile(");
   Trace(fileName);
   Trace(")\n");
   #endif
   
   memcpy(&Settings,&nw->PrintDefaults,sizeof(struct NWPrintSettings));
   
   i=SPOOLCMD-5;
   p=command;
   while(*fileName && i)
   {  while((*fileName && *fileName<'0' || *fileName>'9' && *fileName<'A') && i)
      {  *p++=' ';
         i--;
         fileName++; }

      if(!i || !*fileName)
         break;

      i--;
      *p++='-';

      while((*fileName>='0' && *fileName<='9' || *fileName>='A' || *fileName=='-' || *fileName=='=') && i)
      {  i--;
         if(*fileName=='=')
            *p++=' ';
         else
            *p++=*fileName;
         fileName++;
      }
   }
   *p=0;

   #if defined DEBUG
   Trace("\n Commandline=\"");
   Trace(command);
   Trace("\"\n");
   #endif
   
   if((err=ParseCommand("banner/k,form/k,name/k,queue/k,"
                     #ifdef NDS
                        "printer/k,server/k,"
                     #endif
                        "copies/k/e,tabs/k/e,lines/k/e,columns/k/e,"
                        "ff/s,nobanner=nb/s,noff/s,nonotify=nn/s,notify/s",
                        command,&result,sizeof(result)))!=NULL)
      return err;

#if defined DEBUG
   Trace("\n banner=  "); if(result.banner)  Trace(result.banner);
   Trace("\n form=    "); if(result.form)    Trace(result.form);
   Trace("\n name=    "); if(result.name)    Trace(result.name);
   Trace("\n queue=   "); if(result.queue)   Trace(result.queue);
#ifdef NDS
   Trace("\n printer= "); if(result.printer) Trace(result.printer);
   Trace("\n server=  "); if(result.server)  Trace(result.server);
#endif
   Trace("\n copies=  "); TraceVal((int)result.copies);
   if(result.copies && result.copies->zero==0)
      TraceVal(result.copies->value);
   Trace("\n tabs=    "); TraceVal((int)result.tabs); if(result.tabs) TraceVal(result.tabs->zero);
   if(result.tabs && result.tabs->zero==0)
      TraceVal(result.tabs->value);

   Trace("\n ff=      "); TraceByte(result.ff);
   Trace("\n nobanner="); TraceByte(result.nobanner);
   Trace("\n noff=    "); TraceByte(result.noff);
   Trace("\n nonotify="); TraceByte(result.nonotify);
   Trace("\n notify=  "); TraceByte(result.notify);
#endif

   if(result.banner)
      Settings.Banner=result.banner;
   if(result.form)
      Settings.Form=result.form;
   if(result.name)
      Settings.Name=result.name;
   if(result.queue)
      Settings.Queue=result.queue;
#ifdef NDS
   if(result.printer)
      Settings.Printer=result.printer;
   if(result.server)
      Settings.Server=result.server;
#endif
   if(result.copies && result.copies->zero==0)
      Settings.Copies=result.copies->value;
   if(result.tabs && result.tabs->zero==0)
      Settings.Tabs=result.tabs->value;
   if(result.lines && result.lines->zero==0)
      Settings.Lines=result.lines->value;
   if(result.columns && result.columns->zero==0)
      Settings.Columns=result.columns->value;

   if(result.ff)
      Settings.FF=1;
   if(result.noff)
      Settings.FF=0;
   if(result.nobanner)
   {  Settings.Banner=NULL;
      Settings.Name=NULL; }
   if(result.nonotify)
      Settings.Notify=0;
   if(result.notify)
      Settings.Notify=1;

   if((file=nwclient_spool(&Settings,nw))==NULL)
   {  nw->ReturnStruct[0]=nw->ReturnStruct[1]=0;

      if(-nw->reply.header.CompletionCode==NCP_FILE_NAME_INVALID)
         return &Error_InvalidName;

      return NULL; }

   file->RosHandle=rosHandle;
   file->CurrentSize=0;

   nw->ReturnStruct[0]=(file->Flags&FILE_WRITABLE)?3<<30:1<<30;
   nw->ReturnStruct[1]=(int)file;
   nw->ReturnStruct[2]=file->Server->BlockSize;
   nw->ReturnStruct[3]=file->CurrentSize=nwclient_filesize(file,nw);
   nw->ReturnStruct[4]=(nw->ReturnStruct[3]+nw->ReturnStruct[2]-1)&~(nw->ReturnStruct[2]-1);

   return NULL;
}


/* +---------------------------------------------------------------------+
   |                                                                     |
   |   Highly optimized streaming read.                                  |
   |                                                                     |
   +---------------------------------------------------------------------+ */

#define WINDOWSIZE 4

struct kernel_error *ReadFile(struct NWClient *nw,struct File *file,void *buffer,int numBytes,int fileOffset)
{  int r;
   struct kernel_error *err=&Error_ReadError;
   char sequence,transit;
   struct mbuf *mb;
   struct Server *fs;
   int now,lastreq=0,tries=100;

   #if defined DEBUG
   Trace("\n ReadFile");
   TraceVal(numBytes);
   TraceVal(fileOffset);
   Trace(")");
   #endif

   fs=file->Server;

   if(numBytes<8*fs->BlockSize || fileOffset&(fs->BlockSize-1) ||
      numBytes&(fs->BlockSize-1) || (nw->Flags&NWC_DONT_STREAM))   /* if small or not aligned */
   {
restart:
      #if defined DEBUG && DEBUG&2
      Trace("\n Go Slow");
      #endif

      while(numBytes>0)
      {  r=nwclient_read(file,buffer,fileOffset,numBytes>fs->BlockSize?fs->BlockSize:numBytes,nw);
         if(r==0)
            return err;

         err=NULL;
         buffer=(void *)(((char *)buffer)+r);
         numBytes-=r;
         fileOffset+=r; }
      return 0;
   }
   
   #if defined DEBUG && DEBUG&2
   Trace("\n Stream");
   #endif

	fs->request.header.Function=0x48;   /* read from file */
   memcpy(fs->request.payload.fn_48.Handle,file->Handle,6);
   fs->request.payload.fn_48.Length=fs->BlockSize;

   transit=0;
   sequence=fs->request.header.Sequence;

   while(numBytes)
   {
      now=MonotonicTime();

      if((now-lastreq)>1)
      {  transit=0;
         tries/=2; }

      if(transit<WINDOWSIZE && fileOffset+transit*fs->BlockSize<fileOffset+numBytes)
      {  
         fs->request.header.Sequence=sequence+transit;
         fs->request.payload.fn_48.Offset=fileOffset+transit*fs->BlockSize;

         if(!(fs->request.payload.fn_48.Offset&0xffff))
            ForcePoll();   /* make sure callback handlers get executed now and then */

         #if defined DEBUG && DEBUG&2
         Trace("\n request");
         TraceByte(fs->request.header.Sequence);
         TraceVal(fs->request.payload.fn_48.Offset);
         #endif

         if((r=ipx_send_pack(fs->NcpSocket,&fs->request,"W5bez6bLW",nw->ri))<0)
            Trace("error sending");
         else
         {  lastreq=now;
            transit++; }

         {  volatile int jiffy;
            for(jiffy=0;jiffy<1000;jiffy++)
               {}
         }
      }

      if((mb=ipx_recv_mbuf(fs->NcpSocket,nw->ri))!=NULL)
      {
         #if defined DEBUG && DEBUG&2
         Trace("\n recv");
         #endif

         if((r=UnpackN(mb->next,&nw->reply,"W6bW",NULL,mb->pkthdr.len))>0 &&
             nw->reply.header.Type==NCP_REPLY)
         {
            #if defined DEBUG && DEBUG&2
            Trace(" reply");
            TraceByte(nw->reply.header.Sequence);
            #endif

            if(nw->reply.header.Sequence==sequence)
            {  
               if(nw->reply.header.CompletionCode!=0)
                  return &Error_ReadError;
               
               tries++;

               #if defined DEBUG && DEBUG&2
               Trace("Test length ");
               TraceVal(nw->reply.payload.fn_48.Length);
               TraceVal(fs->BlockSize);
               TraceVal(numBytes);
               #endif

               if(nw->reply.payload.fn_48.Length==fs->BlockSize || numBytes==fs->BlockSize)
               {  
                  int ten=10,
                      len=nw->reply.payload.fn_48.Length;

                  #if defined DEBUG && DEBUG&2
                  Trace(" OK");
                  #endif

                  #if defined DEBUG && DEBUG&2
                  Trace(" memcpy(");
                  TraceVal((int)buffer);
                  TraceVal((int)(mtod(mb->next,char *)+ten));
                  TraceVal((int)(mtod(mb->next,char *)+10));
                  TraceVal(len);
                  TraceVal(mb->next->len-ten-PACKETHEADERSIZE);
                  #endif

                  *((int *)((int)(((char *)buffer)+3)&~3))=len;
                  if((r=UnpackN(mb->next,buffer,"n*b",NULL,mb->pkthdr.len+ten))<0)
                  {
                     #if defined DEBUG && DEBUG&2
                     Trace(" data unpack error");
                     #endif
                     return &Error_ReadError;
                  }
                  else
                  {
                     #if defined DEBUG && DEBUG&2
                     Trace(" increment sequence");
                     #endif
                     transit--;
                     sequence++;
                     fileOffset+=fs->BlockSize;
                     numBytes-=fs->BlockSize;
                     buffer=(void *)(((char *)buffer)+fs->BlockSize);
                  }
               }
            }
            else
               tries/=2;
         }
         nw->ri->mbctl.freem(&nw->ri->mbctl,mb);
      }
      if(!tries)
      {  fs->request.header.Sequence=sequence;
         ForcePoll();   /* make sure callback handlers get executed now and then */
         goto restart; }
   }

   fs->request.header.Sequence=sequence;
   return 0;
}

/* ----------------------------------------------------------------------- */

struct kernel_error *WriteFile(struct NWClient *nw,struct File *file,void *buffer,int numBytes,int fileOffset)
{  int r;
   struct kernel_error *err=&Error_WriteError;
   char sequence,transit;
   struct mbuf *mb;
   struct Server *fs;
   int now,lastreq=0,tries=100;

   #if defined DEBUG
   Trace("\n WriteFile(");
   TraceVal((int)buffer);
   TraceVal(numBytes);
   TraceVal(fileOffset);
   Trace(")");
   #endif

   fs=file->Server;

   if(numBytes<8*fs->BlockSize || fileOffset&(fs->BlockSize-1) || 
      numBytes&(fs->BlockSize-1) || (nw->Flags&NWC_DONT_STREAM))   /* if small or not aligned */
   {
restart:

      #if defined DEBUG && DEBUG&2
      Trace("\n Go Slow");
      #endif

      while(numBytes>0)
      {  r=nwclient_write(file,buffer,fileOffset,numBytes>fs->BlockSize?fs->BlockSize:numBytes,nw);
         if(r==0)
            return &Error_WriteError;

         buffer=(void *)(((char *)buffer)+r);
         numBytes-=r;
         fileOffset+=r;

         if(fileOffset>file->CurrentSize)
            file->CurrentSize=fileOffset;
      }
      return 0;
   }
   
   #if defined DEBUG && DEBUG&2
   Trace("\n Stream");
   #endif

	fs->request.header.Function=0x49;   /* write to file */
   memcpy(fs->request.payload.fn_49.Handle,file->Handle,6);
   fs->request.payload.fn_49.Length=fs->BlockSize;

   transit=0;
   sequence=fs->request.header.Sequence;

   while(numBytes)
   {
      now=MonotonicTime();

      if((now-lastreq)>10)
      {  transit=0;
         tries/=2; }

      if(transit<WINDOWSIZE && fileOffset+transit*fs->BlockSize<fileOffset+numBytes)
      {  
         fs->request.header.Sequence=sequence+transit;
         fs->request.payload.fn_49.Offset=fileOffset+transit*fs->BlockSize;

         if(!(fs->request.payload.fn_48.Offset&0xffff))
            ForcePoll();   /* make sure callback handlers get executed now and then */

         #if defined DEBUG && DEBUG&2
         Trace("\n request");
         TraceByte(fs->request.header.Sequence);
         TraceVal(fs->request.payload.fn_49.Offset);
         #endif

         if((mb=nw->ri->mbctl.alloc(&nw->ri->mbctl,(2+5+1+6+4+2),NULL))==NULL ||
            Pack(mb,&fs->request,"W5bez6bLW")<0 ||
            (mb->next=nw->ri->mbctl.alloc_u(&nw->ri->mbctl,fs->BlockSize,
                                            (void *)(((char *)buffer)+transit*fs->BlockSize)))==NULL ||
            ipx_send_mbuf(fs->NcpSocket,mb,nw->ri))
         {
            if(mb)
               nw->ri->mbctl.freem(&nw->ri->mbctl,mb);

            Trace("error sending");
         }
         else
         {  lastreq=now;
            transit++; }

         {  volatile int jiffy;
            for(jiffy=0;jiffy<1000;jiffy++)
               {}
         }
      }

      if((mb=ipx_recv_mbuf(fs->NcpSocket,nw->ri))!=NULL)
      {
         #if defined DEBUG && DEBUG&2
         Trace("\n recv");
         #endif

         if((r=UnpackN(mb->next,&nw->reply,"W6b",NULL,mb->pkthdr.len))>0 &&
             nw->reply.header.Type==NCP_REPLY)
         {
            #if defined DEBUG && DEBUG&2
            Trace(" reply");
            TraceByte(nw->reply.header.Sequence);
            #endif

            if(nw->reply.header.Sequence==sequence)
            {  
               if(nw->reply.header.CompletionCode!=0)
                  return &Error_WriteError;
               
               #if defined DEBUG && DEBUG&2
               Trace(" OK");
               #endif

               tries++;

               #if defined DEBUG && DEBUG&2
               Trace(" increment sequence");
               #endif
               transit--;
               sequence++;
               fileOffset+=fs->BlockSize;
               numBytes-=fs->BlockSize;
               buffer=(void *)(((char *)buffer)+fs->BlockSize);
            }
            else
               tries/=2;
         }
         nw->ri->mbctl.freem(&nw->ri->mbctl,mb);
      }

      if(!tries)
      {  fs->request.header.Sequence=sequence;
         ForcePoll();   /* make sure callback handlers get executed now and then */
         goto restart; }
   }

   fs->request.header.Sequence=sequence;
   return 0;
}

/* ----------------------------------------------------------------------- */

int FileSize(struct NWClient *nw,struct File *file)
{
   #if defined DEBUG && DEBUG&2
   Trace("\n FileSize");
   #endif

   return file->CurrentSize;
}

/* ----------------------------------------------------------------------- */

struct kernel_error *WriteFileZeros(struct NWClient *nw,struct File *file,int fileOffset,int numBytes)
{  int r;

   #if defined DEBUG
   Trace("\n WriteFileZeros");
   TraceVal(fileOffset);
   TraceVal(numBytes);
   Trace(")\n");
   #endif

   if(numBytes && fileOffset<file->CurrentSize)
   {  char buffer[256];
      memset(buffer,0,256);

      while(numBytes>0)
      {  r=nwclient_write(file,buffer,fileOffset,numBytes>256?256:numBytes,nw);
         if(r==0)
            return &Error_WriteError;

         numBytes-=r;
         fileOffset+=r;
         if(fileOffset>file->CurrentSize)
            file->CurrentSize=fileOffset;
      }
   }
   else
   {  nwclient_write(file,"",fileOffset,0,nw);
      file->CurrentSize=fileOffset; }

   return NULL;
}

/* ----------------------------------------------------------------------- */

struct kernel_error *ReadDateStamp(struct NWClient *nw,struct File *file)
{  
   #if defined DEBUG
   Trace("\n ReadDateStamp");
   #endif

   if(file->Flags&FILE_GOTDATE)
   {  nw->ReturnStruct[2]=file->Is.File.DateLo;
      nw->ReturnStruct[3]=file->Is.File.DateHi; }
   else
      nw->ReturnStruct[2]=nw->ReturnStruct[3]=0;

   return NULL;
}

/* ----------------------------------------------------------------------- */

struct kernel_error *CloseFile(struct NWClient *nw,struct File *file,int stamp0,int stamp1)
{  int r,dateLo,dateHi;
   struct Server *fs;
   struct Mount *mt;
   char flags,*name;

   #if defined DEBUG
   Trace("\n CloseFile(");
   TraceVal(stamp0);
   TraceVal(stamp1);
   #endif

   fs=file->Server;
   mt=file->Mount;

   if(file->Mount && (stamp0 || stamp1))
   {  
      fs->request.header.Function=0x56;
      fs->request.payload.fn_56_02.SubFunction=2;
      fs->request.payload.fn_56_02.Flags=1;
      memcpy(&fs->request.payload.fn_56_02.Is.Handle,&((short *)file->Handle)[1],4);
      fs->request.payload.fn_56_02.Size=2;

      if((stamp0&0xfff00000)==0xfff00000)
      {  if((fs->request.payload.fn_56_02.Contents.Value=(stamp0>>8)&0xfff)==0xfff ||
            fs->request.payload.fn_56_02.Contents.Value==SOFTLINK ||
            fs->request.payload.fn_56_02.Contents.Value==UNIXEXEC)
            fs->request.payload.fn_56_02.Size=0; 
      }
      else
      {  if((stamp0 || stamp1) && (stamp0!=0xdeaddead || stamp1!=0xdeaddead))
         {  fs->request.payload.fn_56_02.Contents.FullType[0]=stamp0;
            fs->request.payload.fn_56_02.Contents.FullType[1]=stamp1;
            fs->request.payload.fn_56_02.Size=8; }
         else
            fs->request.payload.fn_56_02.Contents.Value=stamp0?0xdead:0xffff;
      }
      fs->request.payload.fn_56_02.BytesToCopy=fs->request.payload.fn_56_02.Length=fs->request.payload.fn_56_02.Size;
      fs->request.payload.fn_56_02.Position=fs->request.payload.fn_56_02.Access=0;

      #if defined DEBUG && DEBUG&2
      if(fs->request.payload.fn_56_02.BytesToCopy)
      {  Trace("\n Set type to");
         TraceVal(fs->request.payload.fn_56_02.Contents.Value); }
      else
         Trace("\n Unset type");
      #endif

      if((!mt || (mt->Flags&MOUNT_LONGNAMES)) && ncp_request("bw5lwk\006zkrkikskckoksn*b","3l",NULL,fs)>=0)
      {  fs->request.header.Function=0x56;
         fs->request.payload.fn_56_01.SubFunction=1;
         fs->request.payload.fn_56_01.EAHandle=nw->reply.payload.fn_56_02.EAHandle;
         ncp_request("bzzl",NULL,NULL,fs); }
   }

   dateLo=file->Is.File.DateLo;
   dateHi=file->Is.File.DateHi;
   flags=file->Flags;
   name=strdup(file->Name);

   if((r=nwclient_close(file,nw))==0 && (flags&FILE_WRITABLE))
   {
      fs->request.payload.fn_57_07.InfoMask=0;

      if(mt && ((flags&FILE_GOTDATE) ||
         ((stamp0&0xfff00000)==0xfff00000 || (stamp0&0xffffff00)==0) && stamp0&0xff && stamp1))
      {
         fs->request.payload.fn_57_07.InfoMask=(DM_MODIFY_TIME|DM_MODIFY_DATE);
         memset((char *)&fs->request.payload.fn_57_07.Info,0,sizeof(fs->request.payload.fn_57_07.Info));

         if((stamp0&0xff)||stamp1)
         {  Trace(" stampie");
            RosToNWDate(stamp0&255,stamp1,
                        &fs->request.payload.fn_57_07.Info.modifyDate,
                        &fs->request.payload.fn_57_07.Info.modifyTime,nw->TimeOffset);
         }
         else
         {  Trace(" restorie");
            RosToNWDate(dateHi,dateLo,
                        &fs->request.payload.fn_57_07.Info.modifyDate,
                        &fs->request.payload.fn_57_07.Info.modifyTime,nw->TimeOffset);
         }
      }

      if((stamp0&0xffffff00)==(0xfff00000|(SOFTLINK<<8))) /* soft link */
      {  fs->request.payload.fn_57_07.Info.attributes|=ATTR_SHARED|ATTR_HIDDEN;
         fs->request.payload.fn_57_07.InfoMask=DM_ATTRIBUTES; }
      else if((stamp0&0xffffff00)==(0xfff00000|(UNIXEXEC<<8))) /* unix exec */
      {  fs->request.payload.fn_57_07.Info.attributes|=ATTR_SHARED|ATTR_SYSTEM;
         fs->request.payload.fn_57_07.InfoMask=DM_ATTRIBUTES; }

      if(fs->request.payload.fn_57_07.InfoMask)
      {
         fs->request.header.Function=0x57;
         fs->request.payload.fn_57.SubFunction=0x07;     /* modify info */
         fs->request.payload.fn_57_07.NameSpace=(mt->Flags&MOUNT_LONGNAMES);
         fs->request.payload.fn_57_07.SearchAttributes=0x8006;
         fs->request.payload.fn_57_07.Volume=mt->Volume;
         fs->request.payload.fn_57_07.Directory=mt->Directory;
         fs->request.payload.fn_57_07.HandleFlag=1;      /* have a dirbase */
         fs->request.payload.fn_57_07.PathBytes=make_path(&fs->request.payload.fn_57_07.PathComponents,name);
         r=ncp_request("bbzwl" REQ_MODIFY REQ_PATH,NULL,NULL,fs);
      }
   }

   free(name);

   return NULL;
}

/* ----------------------------------------------------------------------- */

struct kernel_error *WriteCatInfo(struct NWClient *nw,const char *fileName,int stamp0,int stamp1,int attributes)
{  struct Mount *mt;
   struct Server *fs;
   char buffer[FILENAME_MAX];
   int r,notype=0;

   #if defined DEBUG
   Trace("\n WriteCatInfo(");
   Trace(fileName);
   TraceVal(stamp0);
   TraceVal(stamp1);
   TraceVal(attributes);
   #endif

   #if 0
   if(ReadCatInfo(nw,fileName)==NULL && nw->ReturnStruct[0]==2)
      attributes|=(1<<1);  /* never try to make directories read-only */
   #endif

   if((attributes&0x0a)==0)   /* if clearing lock bit, then also set write bit */
      attributes|=1<<1;

   if((mt=RosToNWName(buffer,fileName,nw,0))==NULL)
      return &Error_InvalidName;
   fs=mt->Server;
   mt->Flags|=MOUNT_RESCANDIR;

   fs->request.header.Function=0x57;
   fs->request.payload.fn_57.SubFunction=0x07;     /* modify info */
   fs->request.payload.fn_57_07.NameSpace=(mt->Flags&MOUNT_LONGNAMES);
   fs->request.payload.fn_57_07.SearchAttributes=0x8006;
   if(((stamp0&0xfff00000)==0xfff00000 || (stamp0&0xffffff00)==0) && stamp0&0xff && stamp1)
      fs->request.payload.fn_57_07.InfoMask=(DM_MODIFY_TIME|DM_MODIFY_DATE|DM_ATTRIBUTES);
   else
      fs->request.payload.fn_57_07.InfoMask=DM_ATTRIBUTES;

   memset((char *)&fs->request.payload.fn_57_07.Info,0,sizeof(fs->request.payload.fn_57_07.Info));

   RosToNWDate(stamp0&255,stamp1,
      &fs->request.payload.fn_57_07.Info.modifyDate,
      &fs->request.payload.fn_57_07.Info.modifyTime,nw->TimeOffset);

   fs->request.payload.fn_57_07.Info.attributes=
      ((attributes&(1<<3))?ATTR_RENAMEINHIBIT|ATTR_DELETEINHIBIT:0) |
      ((attributes&((1<<1)|(1<<5)))==0?ATTR_READONLY:0) |
      ((attributes&(3<<4))?ATTR_SHARED:0);

   if((stamp0&0xffffff00)==(0xfff00000|(SOFTLINK<<8))) /* soft link */
   {  notype=1;
      fs->request.payload.fn_57_07.Info.attributes|=ATTR_SHARED|ATTR_HIDDEN; }
   else if((stamp0&0xffffff00)==(0xfff00000|(UNIXEXEC<<8))) /* unix exec */
   {  notype=1;
      fs->request.payload.fn_57_07.Info.attributes|=ATTR_SHARED|ATTR_SYSTEM; }

   fs->request.payload.fn_57_07.Volume=mt->Volume;
   fs->request.payload.fn_57_07.Directory=mt->Directory;
   fs->request.payload.fn_57_07.HandleFlag=1;      /* have a dirbase */
   fs->request.payload.fn_57_07.PathBytes=make_path(&fs->request.payload.fn_57_07.PathComponents,buffer);

   #define REQ_MODIFY "lwwlwwlwwl3wl"

   if((r=ncp_request("bbzwl" REQ_MODIFY REQ_PATH,NULL,NULL,fs))<0)
      return LookupError(r);

   fs->request.header.Function=0x57;
   fs->request.payload.fn_57.SubFunction=0x06;     /* get info */
   fs->request.payload.fn_57_06.NameSpace= /* namespace (0 for DOS or 4 for OS/2 */
   fs->request.payload.fn_57_06.DestNameSpace=(mt->Flags&MOUNT_LONGNAMES);
   fs->request.payload.fn_57_06.SearchAttributes=0x8006;
   fs->request.payload.fn_57_06.RIM=RIM_ALL;
   fs->request.payload.fn_57_06.Volume=mt->Volume;
   fs->request.payload.fn_57_06.Directory=mt->Directory;
   fs->request.payload.fn_57_06.HandleFlag=1;      /* have a dirbase */
   fs->request.payload.fn_57_06.PathBytes=make_path(&fs->request.payload.fn_57_06.PathComponents,buffer);
   if((r=ncp_request("3bwl"REQ_PATH,ENTRY_INFO,buffer,fs))>=0)
   {
      if(!notype)
      {
         fs->request.header.Function=0x56;
         fs->request.payload.fn_56_02.SubFunction=2;
         fs->request.payload.fn_56_02.Flags=0;
         fs->request.payload.fn_56_02.Is.Entry.Volume=nw->reply.payload.fn_57_06.VolNumber;
         fs->request.payload.fn_56_02.Is.Entry.DirEntry=nw->reply.payload.fn_57_06.DosDirNum;
         fs->request.payload.fn_56_02.Size=2;

         if((stamp0&0xfff00000)==0xfff00000)
         {  if((fs->request.payload.fn_56_02.Contents.Value=(stamp0>>8)&0xfff)==0xfff)
            fs->request.payload.fn_56_02.Size=0; }
         else
         {  if((stamp0 || stamp1) && (stamp0!=0xdeaddead || stamp1!=0xdeaddead))
            {  fs->request.payload.fn_56_02.Contents.FullType[0]=stamp0;
               fs->request.payload.fn_56_02.Contents.FullType[1]=stamp1;
               fs->request.payload.fn_56_02.Size=8; }
            else
               fs->request.payload.fn_56_02.Contents.Value=stamp0?0xdead:0xffff;
         }
         fs->request.payload.fn_56_02.BytesToCopy=fs->request.payload.fn_56_02.Length=fs->request.payload.fn_56_02.Size;
         fs->request.payload.fn_56_02.Position=fs->request.payload.fn_56_02.Access=0;

         #if defined DEBUG && DEBUG&2
         if(fs->request.payload.fn_56_02.BytesToCopy)
         {  Trace("\n Set type to");
            TraceVal(fs->request.payload.fn_56_02.Contents.Value); }
         else
            Trace("\n Unset type");
         #endif

         if((mt->Flags&MOUNT_LONGNAMES) && ncp_request("bw5lwk\006zkrkikskckoksn*b","3l",NULL,fs)>=0)
         {  fs->request.header.Function=0x56;
            fs->request.payload.fn_56_02.SubFunction=1;
            fs->request.payload.fn_56_01.EAHandle=nw->reply.payload.fn_56_02.EAHandle;
            ncp_request("bzzl",NULL,NULL,fs); }
      }
   }

   return NULL;
}

/* ----------------------------------------------------------------------- */

struct kernel_error *DeleteFile(struct NWClient *nw,const char *fileName)
{  char buffer[FILENAME_MAX];
   struct Mount *mt;
   struct Server *fs;
   int r;

   #if defined DEBUG
   Trace("\n DeleteFile");
   #endif

   if(ReadCatInfo(nw,fileName))     /* prepare the return values */
      nw->ReturnStruct[0]=nw->ReturnStruct[2]=nw->ReturnStruct[3]=nw->ReturnStruct[4]=nw->ReturnStruct[5]=0;

   #if defined DEBUG && DEBUG&2
   Trace("\n now really going to delete\n");
   #endif

   if((mt=RosToNWName(buffer,fileName,nw,0))==NULL)
      return &Error_InvalidName;
   fs=mt->Server;
   mt->Flags|=MOUNT_RESCANDIR;

   fs->request.header.Function=0x57;
   fs->request.payload.fn_57.SubFunction=0x08;     /* delete file */
   fs->request.payload.fn_57_08.NameSpace=(mt->Flags&MOUNT_LONGNAMES); 
   fs->request.payload.fn_57_08.SearchAttributes=0x8006;
   fs->request.payload.fn_57_08.Volume=mt->Volume;
   fs->request.payload.fn_57_08.Directory=mt->Directory;
   fs->request.payload.fn_57_08.HandleFlag=1;      /* have a dirbase */
   fs->request.payload.fn_57_08.PathBytes=make_path(&fs->request.payload.fn_57_08.PathComponents,buffer);
   if((r=ncp_request("bbzw"REQ_PATH,NULL,NULL,fs))<0)
   {  nw->ReturnStruct[0]=0;
      #if defined DEBUG && DEBUG&2
      ShowError("delete=",r);
      #endif
      if(r==NCP_NO_FILES_FOUND)
      {  
         #if defined DEBUG && DEBUG&2
         Trace("not found is not an error ?!");
         #endif
         return NULL;
      }
      return &Error_Locked; 
   }

   return NULL;
}

/* ----------------------------------------------------------------------- */

struct kernel_error *CreateFile(struct NWClient *nw,const char *fileName,int stamp0,int stamp1,int fileSize)
{  struct File *file;
   struct Mount *mt;
   struct Server *fs;
   char buffer[FILENAME_MAX];
   struct kernel_error *err;

   #if defined DEBUG
   Trace("\n CreateFile(");
   Trace(fileName);
   TraceVal(fileSize);
   TraceVal(stamp0);
   TraceVal(stamp1);
   Trace(")\n");
   #endif

   if((mt=RosToNWName(buffer,fileName,nw,0))==NULL)
      return &Error_InvalidName;
   fs=mt->Server;

   if((file=nwclient_open(mt,buffer,O_CREATE|O_REPLACE|O_RDWR,nw))==NULL)
   {  nw->ReturnStruct[1]=0;
      #if defined DEBUG && DEBUG&4
      Trace(" Create:cant open ");
      #endif
      return &Error_FileOpen; }

   err=WriteFileZeros(nw,file,fileSize,0);
   CloseFile(nw,file,stamp0,stamp1);

   #if defined DEBUG && DEBUG&2
   Trace(" Created");
   TraceVal((int)err);
   #endif

   return err;
}

/* ----------------------------------------------------------------------- */

struct kernel_error *CreateDirectory(struct NWClient *nw,const char *dirName)
{  struct Mount *mt;
   char buffer[FILENAME_MAX];
   struct File *file;
   
   #if defined DEBUG
   Trace("\n CreateDirectory(");
   Trace(dirName);
   Trace(")\n");
   #endif

   if((mt=RosToNWName(buffer,dirName,nw,0))==NULL)
      return &Error_InvalidName;
   
   if((file=nwclient_open(mt,buffer,O_CREATE | (ATTR_DIR<<17) | (255<<8),nw))!=NULL)
      nwclient_close(file,nw);

   #if defined DEBUG && DEBUG&2
   Trace("\n created");
   #endif

   return NULL;
}

/* ----------------------------------------------------------------------- */

int RenameFile(struct NWClient *nw,const char *oldFileName,const char *newFileName)
{  struct Mount *mt;
   struct Server *fs;
   char srcbuffer[FILENAME_MAX],dstbuffer[FILENAME_MAX],buffer[FILENAME_MAX];

   #if defined DEBUG
   Trace("\n RenameFile");
   Trace(oldFileName);
   Trace(" as ");
   Trace(newFileName);
   #endif

   if((mt=RosToNWName(buffer,oldFileName,nw,0))==NULL)
      return (int)&Error_InvalidName;

   fs=mt->Server;
   mt->Flags|=MOUNT_RESCANDIR;

   fs->request.payload.fn_57_04.SourcePath=srcbuffer+1;
   fs->request.payload.fn_57_04.SourcePathBytes=make_path(srcbuffer,buffer)-1;
   fs->request.payload.fn_57_04.SourcePathComponents=*srcbuffer;
   if(mt!=RosToNWName(buffer,newFileName,nw,0))
      return (int)&Error_BadRename;
   fs->request.payload.fn_57_04.DestPath=dstbuffer+1;
   fs->request.payload.fn_57_04.DestPathBytes=make_path(dstbuffer,buffer)-1;
   fs->request.payload.fn_57_04.DestPathComponents=*dstbuffer;
   
   fs->request.header.Function=0x57;
   fs->request.payload.fn_57.SubFunction=0x04;     /* rename file or directory */
   fs->request.payload.fn_57_04.NameSpace=(mt->Flags&MOUNT_LONGNAMES); 
   fs->request.payload.fn_57_04.Flags=1;           /* 'make it work' flag */
   fs->request.payload.fn_57_04.SearchAttributes=0x8006;
   fs->request.payload.fn_57_04.DestVolume=fs->request.payload.fn_57_04.SourceVolume=mt->Volume;
   fs->request.payload.fn_57_04.DestDirectory=fs->request.payload.fn_57_04.SourceDirectory=mt->Directory;
   fs->request.payload.fn_57_04.DestHandleFlag=fs->request.payload.fn_57_04.SourceHandleFlag=1;      /* have a dirbase */

   if(ncp_request("3bw" "blbb" "blbb" "n*sn*s",NULL,NULL,fs)<0)
   {  fs->request.payload.fn_57_04.SearchAttributes=0x8016;
      if(ncp_request("3bw" "blbb" "blbb" "n*sn*s",NULL,NULL,fs)<0)
         return (int)&Error_BadRename;
   }

   return 0;
}

/* ----------------------------------------------------------------------- */

void Shutdown(struct NWClient *nw)
{
   #if defined DEBUG
   Trace("\n Shutdown");
   #endif

   StarDetach("-a",0,nw);
}

/* ----------------------------------------------------------------------- */

struct kernel_error *FreeSpace(struct NWClient *nw,const char *fileName)
{  char buffer[FILENAME_MAX];
   struct Mount *mt;
   struct Server *fs;
   int i,r;
  
   #if defined DEBUG
   Trace("\n FreeSpace");
   Trace(fileName);
   #endif

   if((mt=RosToNWName(buffer,fileName,nw,0))==NULL)
      return &Error_InvalidName;
   fs=mt->Server;

   nw->ReturnStruct[0]=nw->ReturnStruct[1]=nw->ReturnStruct[2]=1<<31-1;

   fs->request.header.Function=0x16;
   fs->request.payload.fn_16.SubFunction=0x33;
   fs->request.payload.fn_16_33.Length=5;
   fs->request.payload.fn_16_33.Volume=mt->Volume;

   if((r=ncp_request(REQ_16"l","w8l",NULL,fs))==NCP_NCP_NOT_SUPPORTED)
   {
      fs->request.header.Function=0x16;
      fs->request.payload.fn_16.SubFunction=0x2c;
      fs->request.payload.fn_16_2c.Length=1;
      fs->request.payload.fn_16_2c.Volume=mt->Volume;

      if((r=ncp_request(REQ_16"b","6l4zbc",buffer,fs))==NCP_NCP_NOT_SUPPORTED)
      {  fs->request.header.Function=0x12;
         fs->request.payload.fn_12.Volume=mt->Volume;
         if((r=ncp_request("b","5w16bw",NULL,fs))==0)
         {
            nw->ReturnStruct[3]=0;  /* return values are in bytes */
            nw->ReturnStruct[0]=nw->reply.payload.fn_12.FreeBlocks*
                                nw->reply.payload.fn_12.SectorsPerBlock;
            nw->ReturnStruct[2]=nw->reply.payload.fn_12.TotalBlocks*
                                nw->reply.payload.fn_12.SectorsPerBlock;
         }
      }
      else
      {
         nw->ReturnStruct[3]=9;  /* number of bits to shift by */
         
         nw->ReturnStruct[0]=(nw->reply.payload.fn_16_2c.FreeBlocks+
                              nw->reply.payload.fn_16_2c.PurgableBlocks+
                              nw->reply.payload.fn_16_2c.NonPurgableBlocks)*
                             nw->reply.payload.fn_16_2c.SectorsPerBlock;
         nw->ReturnStruct[2]=nw->reply.payload.fn_16_2c.TotalBlocks*
                             nw->reply.payload.fn_16_2c.SectorsPerBlock;
      }
   }
   else
   {
      for(i=0;(1<<i)<nw->reply.payload.fn_16_33.SectorSize;i++)
         {}
      nw->ReturnStruct[3]=i;  /* number of bits to shift by */
      
      nw->ReturnStruct[0]=nw->reply.payload.fn_16_33.FreeLimboSectors +
                           (nw->reply.payload.fn_16_33.FreeBlocks+
                            nw->reply.payload.fn_16_33.FreeSubAllocBlocks)*
                            nw->reply.payload.fn_16_33.SectorsPerBlock;
      nw->ReturnStruct[2]=nw->reply.payload.fn_16_33.TotalBlocks*
                          nw->reply.payload.fn_16_33.SectorsPerBlock;
   }

   return LookupError(r);
}

int FreeSpace2(register struct NWClient *nw,int dummy,unsigned int *result,const char *volName)
{  char buffer[FILENAME_MAX];
   int i;

   #if defined DEBUG
   Trace("\n FreeSpace2(");
   Trace(volName);
   #endif

   strcpy(buffer,volName);
   strcpy(buffer+strlen(buffer),".$");

   if(FreeSpace(nw,buffer))
      return 4;

   result[0]=nw->ReturnStruct[2];
   result[2]=nw->ReturnStruct[0];
   result[4]=result[0]-result[2];
   result[1]=result[3]=result[5]=0;

   i=32-nw->ReturnStruct[3];
   result[1]=result[0]>>i;
   result[0]<<=nw->ReturnStruct[3];
   result[3]=result[2]>>i;
   result[2]<<=nw->ReturnStruct[3];
   result[5]=result[4]>>i;
   result[4]<<=nw->ReturnStruct[3];

   return 0;
}

/* +---------------------------------------------------------------------+
   |                                                                     |
   |   Fucked up current directory handling.                             |
   |                                                                     |
   +---------------------------------------------------------------------+ */

struct kernel_error *SetCurrentDir(struct NWClient *nw,const char *dirName)
{  char buffer[FILENAME_MAX];
   struct Mount *mt;
  
   #if defined DEBUG
   Trace("\n SetCurrentDir ");
   Trace(dirName);
   #endif

   if((mt=RosToNWName(buffer,dirName,nw,0))==NULL)
      return &Error_InvalidName;

   nw->CurrentMount=mt;

   return NULL;
}

/* ----------------------------------------------------------------------- */

struct kernel_error *ReadBootOption(struct NWClient *nw,char *buffer)
{  
   #if defined DEBUG
   Trace("\n ReadBootOption ");
   #endif

   if(nw->CurrentMount)
   {  buffer[0]=strlen(nw->CurrentMount->LocalName);
      strcpy(buffer+1,nw->CurrentMount->LocalName); }
   else
      strcpy(buffer,"\005Unset");

   #if defined DEBUG
   Trace(buffer);
   #endif

   return NULL;
}


/* ----- EOF fsmain.c ----- */

