/* #------------------------------------------------------------------------#
   |                                                                        |
   |   FILER.C                                                              |
   |                                                                        |
   |   Iconbar front-end (i.e. filer) for NWClient.                         |
   |                                                                        |
   |   Copyright 1999, Frank A. Vorstenbosch                                |
   |                                                                        |
   #------------------------------------------------------------------------# */

/* $Id: filer.c,v 1.9 2001/12/22 12:19:50 frank Exp $ */

#include "filer.h"
#include "errors.h"

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Constant data and function prototypes.                               |
   |                                                                        |
   +------------------------------------------------------------------------+ */

#define DIRNAME "<NWFiler$Dir>."

static char SpriteFileName[]=DIRNAME "Sprites";
static char TemplateFileName[]=DIRNAME "Template";
static char MessageFileName[]=DIRNAME "Messages";
static char OurName[]="NWClient Filer";
static char LoginBoot[]="wimptask run nw::Login.$.!Boot";
   #define LOGINBOOTOFFSET 13    /* offset to path name */
   #define LOGINBOOTOFFSET2 24   /* offset to leaf name */


static void FreeMemory(register struct Filer *nf);


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Init and exit functions.                                             |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct Filer *filer_init(void)
{  struct Filer *nf;
   struct kernel_error *err;

   Trace("\n filer_init() {");

   if((err=AllocMem(sizeof(struct Filer),(int **)&nf))!=NULL)
   {  if(!err->errorcode)
         err->errorcode=-1;
      return (struct Filer *)err; }
   memset((char *)nf,0,sizeof(struct Filer));

   NewList(&nf->Icons);
   filer_modechange(nf);

   Trace("}");
   return nf;
}

/* -------------------------------------------------------------------------- */

void filer_exit(register struct Filer *nf)
{
   Trace("\n filer_exit(");
   TraceVal((int)nf);
   Trace(") {");

   if(nf->TaskHandle)
      Wimp_CloseDown(nf->TaskHandle,TASK);

   FreeMemory(nf);
   free(nf);

   Trace("}");
}

/* -------------------------------------------------------------------------- */

void filer_servicecall(int r0,register struct Filer *nf,int r2)
{
   nf->ServiceR0=r0;
   nf->ServiceR2=r2;
}

/* -------------------------------------------------------------------------- */

void filer_modechange(register struct Filer *nf)
{  int i;

   Trace("\n filer_modechange xs,ys,w,h=");

   ReadModeVariable(-1,4,&i); nf->XShift=i;
   ReadModeVariable(-1,5,&i); nf->YShift=i;
   ReadModeVariable(-1,11,&nf->Width);
   ReadModeVariable(-1,12,&nf->Height);

   nf->Width++;
   nf->Height++;

   TraceVal(nf->XShift);
   TraceVal(nf->YShift);
   TraceVal(nf->Width);
   TraceVal(nf->Height);
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   The remainder of this module is run in USR mode.                     |
   |                                                                        |
   +------------------------------------------------------------------------+ */

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Utility subroutines.                                                 |
   |                                                                        |
   +------------------------------------------------------------------------+ */

static char *strdup(const char *string)
{  char *p;

   if(AllocMem(sizeof(struct Icon),(int **)&p))
      return NULL;

   strcpy(p,string);
   return p;
}


/* -------------------------------------------------------------------------- */

static void FreeIcon(struct Icon *icon)
{  int block[2];

   block[0]=-2;
   block[1]=icon->Icon;
   Wimp_DeleteIcon(block);

   free((void *)icon->Text);
   RemoveNode(icon);
   free(icon);
}


/* -------------------------------------------------------------------------- */

static void FreeIconList(register struct Filer *nf)
{  struct Icon *icon,*next;

   forList2(&nf->Icons,icon,next)
      FreeIcon(icon);
}


/* -------------------------------------------------------------------------- */
   
static void FreeMemory(register struct Filer *nf)
{  
   FreeIconList(nf);
   
   if(nf->InfoTemplate)
   {  free(nf->InfoTemplate);
      nf->InfoTemplate=NULL; }

   if(nf->LoginTemplate)
   {  free(nf->LoginTemplate);
      nf->LoginTemplate=NULL; }

   if(nf->MountTemplate)
   {  free(nf->MountTemplate);
      nf->MountTemplate=NULL; }

   if(nf->AttachTemplate)
   {  free(nf->AttachTemplate);
      nf->AttachTemplate=NULL; }

   if(nf->Messages)
   {  MessageTrans_CloseFile(nf->Messages);
      nf->Messages=NULL; }
}


/* -------------------------------------------------------------------------- */

int ModuleVersion(const char *name)
{  int i;

   if((i=SwiNumberFromString(name))==0)
      return -1;

   if(CallASwi(i,&i,NULL,NULL))
      return 0;

   return i;
}

/* -------------------------------------------------------------------------- */

void CatVersion(char *dest,const char *name)
{  int i;
   char s[6];

   i=ModuleVersion(name);
   
   if(i<0)
      strcat(dest,"?");
   else
   {
      s[0]=' ';
      s[1]='0'+(i/100)%10;
      s[2]='.';
      s[3]='0'+(i/10)%10;
      s[4]='0'+i%10;
      s[5]=0;
      strcat(dest,s);
   }
}

/* -------------------------------------------------------------------------- */

struct kernel_error *OpenWindow(register struct Filer *nf,int *window,int *template,int icon)
{  int block[10];
   struct kernel_error *err;

   Trace("\n OpenWindow(");
   TraceVal((int)window);
   TraceVal((int)template);
   TraceVal(icon);
   Trace(") {");

   block[0]=(int)window;
   if((err=Wimp_GetWindowState(block))!=NULL)
      return err;

   block[3]-=block[1];  /* calculate window size */
   block[4]-=block[2];

   block[1]=((nf->Width<<nf->XShift)-block[3])/2;    /* top-left position */
   block[2]=((nf->Height<<nf->YShift)-block[4])/2;

   block[3]+=block[1];  /* set bottom-right position */
   block[4]+=block[2];

   block[7]=-1;

   if((err=Wimp_OpenWindow(block))==NULL)
   {
      if(icon>=0)
         err=Wimp_SetCaretPosition(window,icon,strlen((char *)(template[22+icon*8+5])));
   }

   Trace(" }");
   return err;
}


/* -------------------------------------------------------------------------- */

struct kernel_error *CloseWindow(int *window)
{  
   Wimp_CreateMenu((struct Menu *)-1,0,0);
   Wimp_CloseWindow((int *)&window);
   
   return NULL;
}


/* -------------------------------------------------------------------------- */

int IconState(int *window,int icon)
{  int block[10];

   block[0]=(int)window;
   block[1]=icon;
   if(Wimp_GetIconState(block)!=NULL)
      return 0;
   if(block[6]&(1<<21))
      return 1;

   return 0;
}


/* -------------------------------------------------------------------------- */

void ShowFree(register struct Filer *nf,struct Icon *icon)
{
   strcpy((char *)(nf->Buffer),"showfree -fs nw ");
   strcat((char *)(nf->Buffer),icon->Text);
   OSCLI((char *)(nf->Buffer));
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   These routines make the icons on the icon bar.                       |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct kernel_error *MakeIcon(register struct Filer *nf,int position,int mode,const char *name,struct Icon **icon)
{  int w,h;
   struct kernel_error *err;
   char *sprite="Snwgrey";

   if(mode>=MODE_MOUNT)
      sprite="Snwred";

   if((err=AllocMem(sizeof(struct Icon),(int **)icon))!=NULL)
      return err;

   memset((char *)(*icon),0,sizeof(struct Icon));

   if((err=GetSpriteSize(sprite+1,NULL,&w,&h))!=NULL)
   {  free(icon);
      return err; }

   (*icon)->Window=-6;
   (*icon)->MinX=0;
   (*icon)->MinY=-16;
   (*icon)->MaxX=2*w;
   (*icon)->MaxY=2*h+36-16;
   (*icon)->Flags=0x1700610B;
   (*icon)->Text=name;
   (*icon)->Sprite=sprite;
   (*icon)->Length=strlen((*icon)->Sprite);
   (*icon)->Mode=mode;
   (*icon)->IconPosition=position;

   if((err=Wimp_CreateIcon(0x58000010-position,&(*icon)->Window,&(*icon)->Icon))!=NULL)
   {  free(icon);
      return err; }

   return NULL;
}


/* -------------------------------------------------------------------------- */

#pragma On(Check_ltorg)

struct kernel_error *BuildIcons(register struct Filer *nf)
{  int i,pos,version,serverid,flags,count,mt;
   struct Icon *icon=NULL,*next;
   struct kernel_error *err;
   struct Server
   {  NODE;
      int Handle,
          Flags,
          Mode;
      char *Name,
           *User;
   } *server=NULL;
   LIST Servers;
   LIST OldIcons;
   char *local,*remote;

   NewList(&Servers);

   NewList(&OldIcons);
   forList2(&nf->Icons,icon,next)
   {  RemoveNode(icon);
      AddTail(&OldIcons,icon); }

   nf->Flags&=~FILER_MORESERVERS;

   if((version=ModuleVersion("NWClient_63"))!=-1)
   {
      /* ----- build server list -------------------------------------------------- */

      count=0;

      if((err=AllocMem(sizeof(struct Server),(int **)&server))!=NULL)
         goto abort;

      server->Handle=0;
      err=ServerList(&server->Handle,&server->Name,&server->User,&server->Flags);

      if(err)
      {  if(err->errorcode==0x1e6)  /* SWI not supported */
         {  /* invent one server node for convenience */
            Trace(" serverlist swi not supported");
            server->Name=server->User=NULL;
            server->Handle=0;          /* this is not a real server */

            if(Mount("$","sys:",1,&mt)==NULL)
            {  server->Name=strdup(Lookup(nf->Messages,tokenServer));
               Dismount(mt);
               server->Mode=MODE_MOUNT; }
            else
            {  server->Name=strdup(Lookup(nf->Messages,tokenLogin));
               server->Mode=MODE_LOGIN; }
            AddHead(&Servers,server);
         }
         else
            goto abort;
      }
      else
      {  if(server->Handle)
         {  while(server->Handle)
            {  server->Name=strdup(server->Name);
               strlower(server->Name);

               if(server->User)
                  server->Mode=MODE_MOUNT;
               else
                  server->Mode=MODE_LOGIN;

               i=server->Handle;
               AddHead(&Servers,server);
               count++;

               if((err=AllocMem(sizeof(struct Server),(int **)&server))!=NULL)
                  goto abort;

               server->Handle=i;
               if((err=ServerList(&server->Handle,&server->Name,&server->User,&server->Flags))!=NULL)
                  goto abort;
            }
         }
         free(server);
      }
      if(count>1)
         nf->Flags|=FILER_MORESERVERS;

      /* ----- first do mounted directories --------------------------------------- */

      pos=0;

      mt=0;
      if((err=MountList(&mt,&local,&remote,&flags,&count,&serverid))!=NULL)
         goto abort;

      while(mt)
      {  if((flags&0x81)==0)
         {  forList(&Servers,server)
            {  if(serverid==server->Handle || server->Handle==0)
                  server->Mode=MODE_OPEN; }

            forList(&OldIcons,icon)
            {  if(icon->Mode==MODE_OPEN && mt==icon->Mount &&
                  serverid==icon->Server && pos==icon->IconPosition)
               {  RemoveNode(icon);
                  AddTail(&nf->Icons,icon);
                  icon->MountCount=count;
                  icon->MountFlags=flags;
                  count=-1;
                  break; }
            }

            if(count!=-1)
            {  if((err=MakeIcon(nf,pos,MODE_OPEN,strdup(local),&icon))!=NULL)
                  goto abort;

               icon->Mount=mt;
               icon->MountCount=count;
               icon->MountFlags=flags;
               icon->MountPoint=remote;
               icon->Server=serverid;

               AddTail(&nf->Icons,icon);
            }
            pos++;
         }

         if((err=MountList(&mt,&local,&remote,&flags,&count,&serverid))!=NULL)
            goto abort;
      }

      /* ----- then all servers without any mounts -------------------------------- */

      forList(&Servers,server)
      {  if(server->Mode!=MODE_OPEN && server->Mode!=MODE_ATTACH)
         {
            flags=1;

            forList(&OldIcons,icon)
            {  
               if(server->Mode==icon->Mode && server->Handle==icon->Server &&
                  pos==icon->IconPosition)
               {  RemoveNode(icon);
                  AddTail(&nf->Icons,icon);
                  flags=0;
                  break; }
            }

            if(flags)
            {  if((err=MakeIcon(nf,pos,server->Mode,strdup(server->Name),&icon))!=NULL)
                  goto abort;

               icon->Server=server->Handle;

               AddTail(&nf->Icons,icon);
            }
            pos++;
         }
      }

      /* ----- and finally we do the attach-only icon ----------------------------- */

      if(!pos) /* wot? no icons created? */
      {  flags=1;
         forList(&OldIcons,icon)
         {  if(icon->Mode==MODE_ATTACH)
            {  RemoveNode(icon);
               AddTail(&nf->Icons,icon);
               flags=0;
               break; }
         }

         if(flags)
         {  if((err=MakeIcon(nf,pos,MODE_ATTACH,strdup(Lookup(nf->Messages,tokenAttach)),&icon))!=NULL)
               goto abort;

            AddTail(&nf->Icons,icon);
         }
      }
   }

   err=NULL;
   server=NULL;

abort:
   if(server)
      free(server);

   forList2(&OldIcons,icon,next)
      FreeIcon(icon);

   forList2(&Servers,server,(struct Server *)next)
   {  free(server->Name);
      RemoveNode(server);
      free(server); }

   return err;
}

#pragma Pop(Check_ltorg)


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Setup code, loading sprites, templates, messages.                    |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct kernel_error *LoadTemplate(register struct Filer *nf,const char *name,int **template,int **window)
{  int tsize,wsize;
   struct kernel_error *err;

   Trace("\n LoadTemplate(");
   Trace(name);
   Trace(")");

   if((err=Wimp_LoadTemplate((int *)0,NULL,NULL,(char *)-1,name,0,&tsize,&wsize))!=NULL ||
      (err=AllocMem(((tsize+3)&~3)+((wsize+3)&~3),template))!=NULL ||
      (err=Wimp_LoadTemplate(*template,
                            (int *)((char *)(*template)+((tsize+3)&~3)),
                            (int *)((char *)(*template)+((tsize+3)&~3)+((wsize+3)&~3)),
                            (char *)-1,name,0,&tsize,&wsize))!=NULL)
      return err;

   (*template)[6]=-1;
   (*template)[7]|=2;
   err=Wimp_CreateWindow(*template,window);

   #if defined DEBUG
   if(err)
      Trace(err->message);
   #endif

   return err;
}


/* -------------------------------------------------------------------------- */

struct kernel_error *Setup(register struct Filer *nf)
{  struct kernel_error *err;

   Trace("\n Setup() {");

   #if 0
   if((err=SpriteOp(11,NULL,SpriteFileName))!=NULL ||
      (err=Wimp_OpenTemplate(TemplateFileName))!=NULL)
      return err;
   #else
   if((err=SpriteOp(11,NULL,SpriteFileName))!=NULL)
   {  Trace(" spriteop fails");
      return err; }

   if((err=Wimp_OpenTemplate(TemplateFileName))!=NULL)
   {  Trace(" opentemplate fails");
      return err; }
   #endif

   Trace("<1>");

   err=LoadTemplate(nf,constInfo,  &nf->InfoTemplate,  &nf->InfoWindow  );

   if(!err)
   {  char s[8]={0,0,0,0,0,0,0,0};
      strcpy(indirect(nf->InfoTemplate,1),"NWFiler ");
      memcpy(s,Version,4);
      s[4]=0;
      strcat(indirect(nf->InfoTemplate,1),s);
      err=LoadTemplate(nf,constLogin, &nf->LoginTemplate, &nf->LoginWindow );
   }

   if(!err) err=LoadTemplate(nf,constMount, &nf->MountTemplate, &nf->MountWindow );
   if(!err) err=LoadTemplate(nf,constAttach,&nf->AttachTemplate,&nf->AttachWindow);

   Wimp_CloseTemplate();

   if(MessageTrans_OpenFile(nf->MessageFile,MessageFileName,NULL)==NULL)
      nf->Messages=nf->MessageFile;

   Trace("}");
   return err;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Mount window handling.                                               |
   |                                                                        |
   |   Icon numbers for Mount template.                                     |
   |      0 - OK                                                            |
   |      1 - Cancel                                                        |
   |      2 - Local name                                                    |
   |      3 - Remote name                                                   |
   |      4 - No icon                                                       |
   |      5 - No long file names                                            |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct kernel_error *DoMount(register struct Filer *nf);

struct kernel_error *MountClick(register struct Filer *nf,int icon)
{  struct kernel_error *err=NULL;

   switch(icon)
   {  case 0:  /* OK */
         return DoMount(nf);
      case 1:  /* Cancel */
         return CloseWindow(nf->MountWindow);
      case 4: case 5:
         nf->Buffer[0]=(int)nf->MountWindow;
         nf->Buffer[1]=icon;
         nf->Buffer[2]=1<<21;
         nf->Buffer[3]=0;
         return Wimp_SetIconState(nf->Buffer);
   }
   return NULL;
}

/* -------------------------------------------------------------------------- */

struct kernel_error *MountKey(register struct Filer *nf,int icon,int key)
{  struct kernel_error *err=NULL;

   switch(key)
   {  case 27:  /* Escape */
         return CloseWindow(nf->MountWindow);
      case 13:
         return DoMount(nf);
   }
   return NULL;
}

/* -------------------------------------------------------------------------- */

struct kernel_error *DoMount(register struct Filer *nf)
{  int flags=0;
   struct kernel_error *err=NULL,*err2;

   Trace("\n DoMount() {");
   
   if(IconState(nf->MountWindow,4))
      flags|=1;

   if(IconState(nf->MountWindow,5))
      flags|=4;

   indirect(nf->MountTemplate,2)[strlen(indirect(nf->MountTemplate,2))]=0;
   indirect(nf->MountTemplate,3)[strlen(indirect(nf->MountTemplate,3))]=0;

   err=Mount(indirect(nf->MountTemplate,2),
             indirect(nf->MountTemplate,3),flags,&flags);

   err2=CloseWindow(nf->MountWindow);

   if(err)
   {  char *p;

      Trace("\n Mount error ");
      Trace(err->message);
      TraceVal(err->errorcode);

      nf->Buffer[0]=err->errorcode;
      TraceVal(err->errorcode);

      switch(err->errorcode)
      {
         case NCP_VOL_INVALID:
            p="Em1";
            break;
         case NCP_PATH_INVALID:
            p="Em2";
            break;
         case NCP_FILE_NO_SRCH_PRIV:
            p="Em3";
            break;
         case NCP_PARAMETERS_INVALID:
         default:
            p=NULL;
      }

      if(p)
         p=(char *)Lookup(nf->Messages,p);

      if(p)
      {  strcpy((char *)(nf->Buffer+1),p);
         return (struct kernel_error *)(&nf->Buffer); }

      strcpy((char *)(nf->Buffer+1),Lookup(nf->Messages,"Em0:Error mounting directory:"));
      strcat((char *)(nf->Buffer+1),err->message);
      return (struct kernel_error *)(&nf->Buffer);
   }

   return err2;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Login window handling.                                               |
   |                                                                        |
   |   Icon numbers for Login template.                                     |
   |      0 - OK                                                            |
   |      1 - Cancel                                                        |
   |      2 - User name                                                     |
   |      3 - Password                                                      |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct kernel_error *DoLogin(register struct Filer *nf);

struct kernel_error *LoginClick(register struct Filer *nf,int icon)
{  struct kernel_error *err=NULL;

   switch(icon)
   {  case 0:  /* OK */
         err=DoLogin(nf);
         break;
      case 1:  /* Cancel */
         err=CloseWindow(nf->LoginWindow);
         break;
   }
   return err;
}

/* -------------------------------------------------------------------------- */

struct kernel_error *LoginKey(register struct Filer *nf,int icon,int key)
{  struct kernel_error *err=NULL;

   switch(key)
   {  case 27:  /* Escape */
         err=CloseWindow(nf->LoginWindow);
         break;
      case 13:
         err=DoLogin(nf);
         break;
   }
   return err;
}

/* -------------------------------------------------------------------------- */

struct kernel_error *DoLogin(register struct Filer *nf)
{  struct kernel_error *err=NULL,*err2;
   int h;

   Trace("\n DoLogin() {");

   indirect(nf->LoginTemplate,2)[strlen(indirect(nf->LoginTemplate,2))]=0;
   indirect(nf->LoginTemplate,3)[strlen(indirect(nf->LoginTemplate,3))]=0;

   if(nf->CurrentIcon->Server)
   {  strcpy((char *)(nf->Buffer),nf->CurrentIcon->Text);
      strcat((char *)(nf->Buffer),"/"); }
   else
      *((char *)(nf->Buffer))=0;
   strcat((char *)(nf->Buffer),indirect(nf->LoginTemplate,2));

   err=Login((char *)(nf->Buffer),indirect(nf->LoginTemplate,3));

   if(err)
   {  Trace("\n Login error ");
      Trace(err->message);
      TraceVal(err->errorcode); }

   /* erase password */
   memset(indirect(nf->LoginTemplate,3),0,strlen(indirect(nf->LoginTemplate,3)));

   err2=CloseWindow(nf->LoginWindow);

   if(err->errorcode==NCP_PASSWORD_EXPIRED)
   {  
      char *p;
      if((p=(char *)Lookup(nf->Messages,"El1"))!=NULL)
      {  
         strcpy((char *)(nf->Buffer+1),p);
         nf->Buffer[0]=-1;
         Trace((char *)&nf->Buffer[0]);

         ReportError(err,1,OurName);
      }
      err=NULL;
   }

   if(!err)
   {
      if((h=riscos_open(LoginBoot+LOGINBOOTOFFSET,RO_RDONLY))==0)
      {
Trace(" first riscos_open failed");
         strcpy((char *)nf->Buffer,LoginBoot+LOGINBOOTOFFSET);
         strcat((char *)nf->Buffer,LoginBoot+LOGINBOOTOFFSET2);
         h=riscos_open((char *)nf->Buffer,RO_RDONLY);
Trace(" second riscos_open failed");
      }

      if(h)
      {  riscos_close(h);

         SetVarVal("NW$User",(const char *)indirect(nf->LoginTemplate,2),strlen((const char *)indirect(nf->LoginTemplate,2)),4);
         SetVarVal("NW$Server",nf->CurrentIcon->Text,strlen((const char *)nf->CurrentIcon->Text),4);
         OSCLI(LoginBoot);
         /* OR: Wimp_StartTask (p. 3-177) */
         SetVarVal("NW$User",NULL,-1,4);
         SetVarVal("NW$Server",NULL,-1,4);
      }
      else
         Mount("SYS","SYS:",0,&h);

      *indirect(nf->LoginTemplate,2)=0;
   }
   else
   {  char *p;

      nf->Buffer[0]=err->errorcode;
      TraceVal(err->errorcode);

      switch(err->errorcode)
      {  case NCP_LOGIN_MAX_EXCEEDED:
            p="El2";
            break;
         case NCP_INTRUDER_LOCKOUT:
         case NCP_LOGIN_LOCKOUT:
            p="El3";
            break;
         case NCP_LOGIN_NO_CONN_AVAIL:
            p="El4";
            break;
         case NCP_LOGIN_DISABLED_BY_SUPER:
         case NCP_LOGIN_UNAUTHORIZED_TIME:
            p="El5";
            break;
         case NCP_LOGIN_UNAUTHORIZED_STATION:
            p="El6";
            break;
         case NCP_ACCT_DISABLED:
            p="El7";
            break;
         case NCP_PASSWORD_INVALID:
         case NCP_PATH_NOT_LOCATABLE:
         case NCP_SERVER_UNKNOWN:
            p="El8";
            break;
         default:
            p=NULL;
            break;
      }
      if(p)
         p=(char *)Lookup(nf->Messages,p);

      if(p)
      {  strcpy((char *)(nf->Buffer+1),p);
         return (struct kernel_error *)(&nf->Buffer); }

      strcpy((char *)(nf->Buffer+1),Lookup(nf->Messages,"El0:Error logging in:"));
      strcat((char *)(nf->Buffer+1),err->message);
      return (struct kernel_error *)(&nf->Buffer);
   }
   
   return err?err:err2;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Attach window handling.                                              |
   |                                                                        |
   |   Icon numbers for Attach template.                                    |
   |      0 - OK                                                            |
   |      1 - Cancel                                                        |
   |      2 - Name to attach to                                             |
   |      3 - Server button                                                 |
   |      4 - Tree button                                                   |
   |      5 - Address button                                                |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct kernel_error *DoAttach(register struct Filer *nf);

struct kernel_error *AttachClick(register struct Filer *nf,int icon)
{  struct kernel_error *err=NULL;

   switch(icon)
   {  case 0:  /* OK */
         err=DoAttach(nf);
         break;
      case 1:  /* Cancel */
         err=CloseWindow(nf->AttachWindow);
         break;
   }
   return err;
}

/* -------------------------------------------------------------------------- */

struct kernel_error *AttachKey(register struct Filer *nf,int icon,int key)
{  struct kernel_error *err=NULL;

   switch(key)
   {  case 27:  /* Escape */
         err=CloseWindow(nf->AttachWindow);
         break;
      case 13:
         err=DoAttach(nf);
         break;
   }
   return err;
}

/* -------------------------------------------------------------------------- */

struct kernel_error *DoAttach(register struct Filer *nf)
{  int mode=0;
   struct kernel_error *err=NULL,*err2;

   Trace("\n DoAttach() {");

   if(IconState(nf->AttachWindow,3))
      mode=1;
   else if(IconState(nf->AttachWindow,5))
      mode=2;

   indirect(nf->AttachTemplate,2)[strlen(indirect(nf->AttachTemplate,2))]=0;

   err=Attach(indirect(nf->AttachTemplate,2),mode,&mode);

   err2=CloseWindow(nf->AttachWindow);

   if(err)
   {  char *p;

      Trace("\n Attach error ");
      Trace(err->message);
      TraceVal(err->errorcode);

      nf->Buffer[0]=err->errorcode;
      TraceVal(err->errorcode);

      switch(err->errorcode)
      {
         case NCP_TIMEOUT_FAILURE:
            p="Ea1";
            break;
         case IPX_NO_ROUTE_TO_NETWORK:
            p="Ea2";
            break;
         case IPX_INVALID_ADDRESS:
            p="Ea3";
            break;
         case NCP_PARAMETERS_INVALID:
         default:
            p=NULL;
      }
      if(p)
         p=(char *)Lookup(nf->Messages,p);

      if(p)
      {  strcpy((char *)(nf->Buffer+1),p);
         return (struct kernel_error *)(&nf->Buffer); }

      strcpy((char *)(nf->Buffer+1),Lookup(nf->Messages,"Ea0:Error attaching:"));
      strcat((char *)(nf->Buffer+1),err->message);
      return (struct kernel_error *)(&nf->Buffer);
   }

   return err2;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Menu creation and handling routines.                                 |
   |                                                                        |
   +------------------------------------------------------------------------+ */

void strcpy12(char *dest,const char *source)
{  int i=0;

   while(*source && i<12)
   {  *dest++=*source++;
      i++; }

   while(i<12)
   {  *dest++=0;
      i++; }
}

int BuildMenu(register struct Filer *nf,int mode)
{  int i;

   strcpy(nf->Menu.Name,"NWFiler");
   nf->Menu.Colours=0x00070207;
   nf->Menu.Width=100;
   nf->Menu.Height=44;
   nf->Menu.Gap=0;

   for(i=0;i<8;i++)
   {  nf->Menu.Items[i].Flags=0;
      nf->Menu.Items[i].Handle=0;
      nf->Menu.Items[i].IconFlags=0x7000021; }

   nf->Menu.Items[0].Handle=nf->InfoWindow;
   strcpy12(nf->Menu.Items[0].Name,Lookup(nf->Messages,tokenInfo));
   i=1;

   nf->ThisMenu=mode;

   switch(mode)
   {  case MODE_ATTACH:
         nf->Menu.Items[i].Handle=nf->AttachWindow;
         strcpy12(nf->Menu.Items[i].Name,Lookup(nf->Messages,tokenAttach));
         i++;

         strcpy12(nf->Menu.Items[i].Name,Lookup(nf->Messages,tokenQuit));
         break;

      case MODE_LOGIN:
         nf->Menu.Items[i].Handle=nf->LoginWindow;
         strcpy12(nf->Menu.Items[i].Name,Lookup(nf->Messages,tokenLogin));
         i++;

         nf->Menu.Items[i].Handle=nf->AttachWindow;
         strcpy12(nf->Menu.Items[i].Name,Lookup(nf->Messages,tokenAttach));
         i++;

         strcpy12(nf->Menu.Items[i].Name,Lookup(nf->Messages,tokenDetach));
         i++;

         strcpy12(nf->Menu.Items[i].Name,Lookup(nf->Messages,tokenQuit));
         break;

      case MODE_MOUNT:
         nf->Menu.Items[i].Handle=nf->MountWindow;
         strcpy12(nf->Menu.Items[i].Name,Lookup(nf->Messages,tokenMount));
         i++;

         strcpy12(nf->Menu.Items[i].Name,Lookup(nf->Messages,tokenLogout));
         i++;

         nf->Menu.Items[i].Handle=nf->AttachWindow;
         strcpy12(nf->Menu.Items[i].Name,Lookup(nf->Messages,tokenAttach));
         break;

      case MODE_OPEN:
         nf->Menu.Items[i].Handle=nf->MountWindow;
         strcpy12(nf->Menu.Items[i].Name,Lookup(nf->Messages,tokenMount));
         i++;

         strcpy12(nf->Menu.Items[i].Name,Lookup(nf->Messages,tokenDismount));
         i++;

         strcpy12(nf->Menu.Items[i].Name,Lookup(nf->Messages,tokenLogout));
         i++;

         nf->Menu.Items[i].Handle=nf->AttachWindow;
         strcpy12(nf->Menu.Items[i].Name,Lookup(nf->Messages,tokenAttach));
         i++;

         strcpy12(nf->Menu.Items[i].Name,Lookup(nf->Messages,tokenFree));
         break;

      default:
         i--;
   }

   nf->Menu.Items[i].Flags|=(1<<7);
   return i+1;
}

struct kernel_error *DecodeMenu(register struct Filer *nf,int *selection)
{  int i;

   i=selection[0];   /* only one menu, no submenus */

   switch(nf->ThisMenu*8+selection[0])
   {  
      case MODE_ATTACH*8+0:
      case MODE_LOGIN*8+0:
      case MODE_MOUNT*8+0:
      case MODE_OPEN*8+0:
         nf->InfoTemplate[7]|=(1<<31)|(1<<25);
         Wimp_CreateWindow(nf->InfoTemplate,&nf->InfoWindow);
         return OpenWindow(nf,nf->InfoWindow,nf->InfoTemplate,-1);

      case MODE_ATTACH*8+1:
      case MODE_LOGIN*8+2:
      case MODE_MOUNT*8+3:
      case MODE_OPEN*8+4:
         return OpenWindow(nf,nf->AttachWindow,nf->AttachTemplate,2);
      case MODE_LOGIN*8+3:
         return Detach(nf->CurrentIcon->Server);
      case MODE_LOGIN*8+1:
         *indirect(nf->LoginTemplate,3)=0;   /* clear out password */

         return OpenWindow(nf,nf->LoginWindow,nf->LoginTemplate,
                           strlen(indirect(nf->LoginTemplate,2))?3:2);
      case MODE_MOUNT*8+2:
      case MODE_OPEN*8+3:
         return Logout(nf->CurrentIcon->Server);
      case MODE_MOUNT*8+1:
      case MODE_OPEN*8+1:
         *indirect(nf->MountTemplate,2)=0;            /* clear out local name */
         if(nf->Flags&FILER_MORESERVERS)
         {  strcpy(indirect(nf->MountTemplate,3),nf->CurrentIcon->Text);
            strcat(indirect(nf->MountTemplate,3),"/SYS:"); }
         else
            strcpy(indirect(nf->MountTemplate,3),"SYS:");
         return OpenWindow(nf,nf->MountWindow,nf->MountTemplate,2);
      case MODE_OPEN*8+2:
         return Dismount(nf->CurrentIcon->Mount);
      case MODE_OPEN*8+5:
         ShowFree(nf,nf->CurrentIcon);
         break;
      case MODE_ATTACH*8+2:
      case MODE_LOGIN*8+4:
         nf->Run=0;
         break;
   }

   return NULL;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Mouse click on one of our icons.                                     |
   |                                                                        |
   +------------------------------------------------------------------------+ */


#pragma On(Check_ltorg)

struct kernel_error *IconClick(register struct Filer *nf,int *block)
{  int i;
   struct kernel_error *err=NULL;
   struct Icon *icon=NULL;

   Trace("\n IconClick");

   forList(&nf->Icons,icon)
      if(icon->Icon==block[4])
         break;

   if(IsTailNode(icon))
      return NULL;
   
   nf->CurrentIcon=icon;

   switch(block[2])
   {  case 4:  /* select button */
         switch(icon->Mode)
         {  case MODE_ATTACH:    /* open attach window */
               err=OpenWindow(nf,nf->AttachWindow,nf->AttachTemplate,2);
               break;
            
            case MODE_LOGIN:     /* open login window */
               *indirect(nf->LoginTemplate,3)=0;   /* clear out password */

               err=OpenWindow(nf,nf->LoginWindow,nf->LoginTemplate,
                              strlen(indirect(nf->LoginTemplate,2))?3:2);
               break;
            
            case MODE_MOUNT:     /* open mount window */
               *indirect(nf->MountTemplate,2)=0;            /* clear out local name */
               if(nf->Flags&FILER_MORESERVERS)
               {  strcpy(indirect(nf->MountTemplate,3),icon->Text);
                  strcat(indirect(nf->MountTemplate,3),"/SYS:"); }
               else
                  strcpy(indirect(nf->MountTemplate,3),"SYS:");
               err=OpenWindow(nf,nf->MountWindow,nf->MountTemplate,2);
               break;

            case MODE_OPEN:      /* open directory */
               strcpy((char *)(nf->Buffer),"filer_opendir nw::");
               strcat((char *)(nf->Buffer),icon->Text);
               strcat((char *)(nf->Buffer),".$");
               OSCLI((char *)(nf->Buffer));
               break;
         }
         break;

      case 1:  /* adjust button */
         switch(icon->Mode)
         {  case MODE_ATTACH:    /* try attaching to anything at all */
               err=Attach(NULL,0,&i);
               break;
            case MODE_LOGIN:     /* detach */
               err=Detach(icon->Server);
               break;
            case MODE_MOUNT:     /* logout */
               err=Logout(icon->Server);
               break;
            case MODE_OPEN:      /* show free space */
               ShowFree(nf,icon);
               break;
         }
         break;

      case 2:  /* menu button */
         strcpy(indirect(nf->InfoTemplate,4),"NWClient");
         CatVersion(indirect(nf->InfoTemplate,4),"NWClient_63 ");
         strcat(indirect(nf->InfoTemplate,4),"  IPX");
         CatVersion(indirect(nf->InfoTemplate,4),"IPX_63");

         switch(icon->Mode)
         {  case MODE_MOUNT:
               *indirect(nf->MountTemplate,2)=0;  /* clear out local name */
               if(nf->Flags&FILER_MORESERVERS)
               {  strcpy(indirect(nf->MountTemplate,3),icon->Text);
                  strcat(indirect(nf->MountTemplate,3),"/SYS:"); }
               else
                  strcpy(indirect(nf->MountTemplate,3),"SYS:");
               break;
            case MODE_OPEN:
               strcpy(indirect(nf->MountTemplate,2),icon->Text);
               strcpy(indirect(nf->MountTemplate,3),icon->MountPoint);
               break;
         }
         i=BuildMenu(nf,icon->Mode);
         err=Wimp_CreateMenu(&nf->Menu,block[0]-64,96+i*44);
         break;
   }
   return err;
}

#pragma Pop(Check_ltorg)

/* -------------------------------------------------------------------------- */

void Help(register struct Filer *nf,int *block)
{  char *p=NULL;
   struct Icon *icon=NULL;

   Trace("\n Help");
   TraceVal(block[8]);
   TraceVal(block[9]);

   TraceVal((int)&nf->Menu);

   if(block[8]==(int)nf->MountWindow)
      p="Hm0";
   else if(block[8]==(int)nf->LoginWindow)
      p="Hl0";
   else if(block[8]==(int)nf->AttachWindow)
      p="Ha0";
   else if(block[8]==(int)nf->InfoWindow)
      p="Hi0";
   else if(block[8]==-2)
   {  forList(&nf->Icons,icon)
         if(icon->Icon==block[9])
            break;

      if(!IsTailNode(icon))
      {  switch(icon->Mode)
         {  case MODE_ATTACH:    
               p="Hi1";
               break;
            case MODE_LOGIN:     
               p="Hi2";
               break;
            case MODE_MOUNT:     
               p="Hi3";
               break;
            case MODE_OPEN:      
               p="Hi4";
               break;
      }  }
   }
   else
      p="Hi5";
   
   Trace(" ");

   if(p)
   {  Trace(p);

      if((p=(char *)Lookup(nf->Messages,p))!=NULL)
      {  /* send the reply */
         block[0]=256;
         block[3]=block[2];
         block[4]=0x503; /* help reply */
         if(strlen(p)<256-20)
         {  strcpy((char *)(block+5),p);
            Wimp_SendMessage(17,block,block[1],0);
      }  }
   }  
   else
      Trace("dunno");
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Main loop for filer front-end.                                       |
   |                                                                        |
   +------------------------------------------------------------------------+ */

#pragma On(Check_ltorg)

int filer(register struct Filer *nf,const char *commandTail)
{  static int Messages[2]={0x502,0};   /* messages we're interested in */
   struct kernel_error *err;
   struct
   {  int Attach;
      char buffer[100]; } result;

   Trace("\n filer(");
   TraceVal((int)nf);
   TraceVal((int)commandTail);
   Trace("->\"");
   Trace(commandTail);
   Trace("\") {");

   /* ----- initialize --------------------------------------------------------- */

   if((err=ParseCommand("attach/s",commandTail,&result,sizeof(result)))!=NULL)
   {  ReportError(err,1,OurName);
      return 0; }

   if(nf->TaskHandle)
      Wimp_CloseDown(nf->TaskHandle,TASK);
   nf->TaskHandle=Wimp_Initialise(310,TASK,OurName,Messages);

   if((err=Setup(nf))!=NULL)
   {  
nonstart:Trace(" nonstart");
      nf->Run=0;
      ReportError(err,1,OurName); }
   else
   {  nf->Run=50;
      if(result.Attach)
         Attach(NULL,0,&result.Attach);
   }

   /* ----- main loop ---------------------------------------------------------- */

   while(nf->Run)
   {  Trace("\nPoll");
      err=NULL;
      switch(Wimp_PollIdle((1<<4)|(1<<5)|(1<<22),nf->Buffer,MonotonicTime()+nf->Run,&nf->ServiceR0))
      {  case 2:
            Wimp_OpenWindow(nf->Buffer);
            break;

         case 3:
            CloseWindow((int *)(nf->Buffer[0]));
            
            if((int *)(nf->Buffer[0])==nf->InfoWindow)
            {  nf->InfoTemplate[7]&=~((1<<31)|(1<<25));
               Wimp_CreateWindow(nf->InfoTemplate,&nf->InfoWindow); }
            break;

         case 6:
            if(nf->Buffer[3]==-2)   /* any button on the iconbar */
               err=IconClick(nf,nf->Buffer);
            else
               if(nf->Buffer[2]==4)  /* select button on a window */
               {  
                  if(nf->Buffer[3]==(int)nf->MountWindow)
                     err=MountClick(nf,nf->Buffer[4]);
                  else if(nf->Buffer[3]==(int)nf->LoginWindow)
                     err=LoginClick(nf,nf->Buffer[4]);
                  else if(nf->Buffer[3]==(int)nf->AttachWindow)
                     err=AttachClick(nf,nf->Buffer[4]);
               }
            break;

         case 8:
            if(nf->Buffer[0]==(int)nf->MountWindow)
               err=MountKey(nf,nf->Buffer[1],nf->Buffer[6]);
            else if(nf->Buffer[0]==(int)nf->LoginWindow)
               err=LoginKey(nf,nf->Buffer[1],nf->Buffer[6]);
            else if(nf->Buffer[0]==(int)nf->AttachWindow)
               err=AttachKey(nf,nf->Buffer[1],nf->Buffer[6]);
            break;

         case 9:
            err=DecodeMenu(nf,nf->Buffer);
            break;

         case 13:
            if(nf->ServiceR0==SERVICE_RMKILL)
               nf->Run=50;  /* when module is reloaded, it might be an old one */
            else
               nf->Run=3000;   /* use long interval for PollIdle from now on */

            nf->ServiceR0=0;
            break;

         case 17:
         case 18:
            TraceVal(nf->Buffer[4]);
            switch(nf->Buffer[4])
            {  case 0:        /* quit */
                  nf->Run=0;
                  break;
               case 0x502:    /* help request */
                  Help(nf,nf->Buffer);
                  break;
            }
            break;
      }
      if(nf->Run && !err)
         err=BuildIcons(nf);
      if(err)
      {  nf->Buffer[0]=1;
         strcpy((char *)(nf->Buffer+1),err->message);
         ReportError(err,1,OurName);
      }
   }

   /* ----- close up ----------------------------------------------------------- */

   FreeMemory(nf);

   if(nf->TaskHandle)
   {  Wimp_CloseDown(nf->TaskHandle,TASK);
      nf->TaskHandle=0; }

   Trace("}");
   return 0;
}

/* ----- EOF FILER.C ----- */

