/* #------------------------------------------------------------------------#
   |                                                                        |
   |   fileio.c                                                             |
   |                                                                        |
   |   Low level file I/O for NWClient.                                     |
   |                                                                        |
   |   Copyright 1999-2001, Frank A. Vorstenbosch                           |
   |                                                                        |
   #------------------------------------------------------------------------# */

/* $Id: fileio.c,v 1.7 2001/12/22 12:16:17 frank Exp $ */

#define VERSION 0x00000

#include "nwclient.h"

/* +---------------------------------------------------------------------+
   |                                                                     |
   |   Open a file on the file server.                                   |
   |                                                                     |
   +---------------------------------------------------------------------+ */

struct File *nwclient_open(struct Mount *mt,const char *name,int mode,register struct NWClient *nw)
{  struct File *file;
   struct Server *fs;
   int i,r;
   short handle[3];

   #if defined DEBUG
   Trace("\n nwclient_open");
   #endif

   if(mt)
   {
      fs=mt->Server;

      if(mode&O_CREATE)                /* too aggressive, but that's OK for now */
         mt->Flags|=MOUNT_RESCANDIR;
   }
   else
   {  if((fs=lookup_server(name,&name,nw))==NULL)
         return NULL;
   }

Trace(" in open, fs="); TraceVal((int)fs);

   i=strlen(name);
   if((file=malloc(sizeof(struct File)+i))==NULL)
      return NULL;
   memcpy(file->Name,name,i+1);

   #if defined DEBUG && (DEBUG&2)
   Trace("\nopen/create file:");
   Trace(file->Name);
   #endif

   fs->request.header.Function=0x57;
   fs->request.payload.fn_57.SubFunction=0x01;     /* open/create file or subdir */
   fs->request.payload.fn_57_01.NameSpace=(mt->Flags&MOUNT_LONGNAMES);
   fs->request.payload.fn_57_01.Mode=mode&0xff;
   fs->request.payload.fn_57_01.SearchAttributes=((mode>>17)&ATTR_DIR)?0x8006:6;
   fs->request.payload.fn_57_01.RIM=RIM_ALL;
   fs->request.payload.fn_57_01.CreateAttributes=mode>>17;
   fs->request.payload.fn_57_01.AccessRights=(mode>>8)&0x1ff;
   if(mt)
   {  fs->request.payload.fn_57_01.Volume=mt->Volume;
      fs->request.payload.fn_57_01.Directory=mt->Directory;
      fs->request.payload.fn_57_01.HandleFlag=1; }
   else
   {  fs->request.payload.fn_57_01.Volume=0;
      fs->request.payload.fn_57_01.Directory=0;
      fs->request.payload.fn_57_01.HandleFlag=-1; }
   fs->request.payload.fn_57_01.PathBytes=make_path(&fs->request.payload.fn_57_01.PathComponents,file->Name);
   if((r=ncp_request(REQ_57"bbwllwblbn*b","lb",NULL,fs))<0)
   {  free(file);
      return NULL; }

   memcpy(&handle[1],&nw->reply.payload.fn_57_01.Handle,4);
   handle[0]=handle[1]+1;  /* heavy magic here */
   memcpy(file->Handle,&handle,6);


   file->Flags=(mode&O_WRITE)?FILE_WRITABLE:0;
   file->CurrentSize=-1; /* obviously wrong, but is set in ncpfs2 OpenFile() */

   file->Action=nw->reply.payload.fn_57_01.Action;
   file->Mount=mt;
   file->Server=fs;
   file->DirEntNum=file->VolNumber=-1;

   if(mt)
      AddHead(&mt->Files,file);
   else
      AddHead(&fs->Files,file);

   return file;
}


/* +---------------------------------------------------------------------+
   |                                                                     |
   |   Open a printer spool file on the file server.                     |
   |                                                                     |
   +---------------------------------------------------------------------+ */

struct File *nwclient_spool(struct NWPrintSettings *Settings,register struct NWClient *nw)
{  struct File *file;
   struct Server *fs;
   int r;
   short handle[3];
   struct scan_object *p;

   #if defined DEBUG
   Trace("\n nwclient_spool");
   #endif

   if((fs=lookup_server(Settings->Server,NULL,nw))==NULL)
      return NULL;

   if((r=nwclient_scanobject(NCP_TYPE_PQUEUE,Settings->Queue,-1,&p,fs))<0)
      return NULL;

   if((file=malloc(sizeof(struct File)))==NULL)
      return NULL;
   file->Server=fs;
   
   fs->request.header.Function=0x17;
   fs->request.payload.fn_17_79.Length=0x011d;
   fs->request.payload.fn_17_79.SubFunction=0x79;
   file->Is.PrintJob.Queue=fs->request.payload.fn_17_79.QueueID=p->ObjectID;

   memset((void *)&fs->request.payload.fn_17_79.Entry,0,sizeof(fs->request.payload.fn_17_79.Entry));
   memset((void *)&fs->request.payload.fn_17_79.JobTextDescription,0,50);
   memset((void *)&fs->request.payload.fn_17_79.ClientRecordArea,0,sizeof(fs->request.payload.fn_17_79.ClientRecordArea));

   fs->request.payload.fn_17_79.Entry.TargetServerID=-1;
   memset((void *)&fs->request.payload.fn_17_79.Entry.TargetExecTime,-1,6);

   fs->request.payload.fn_17_79.Entry.JobControlFlags=0x10;
   fs->request.payload.fn_17_79.Entry.FileNameLen=0;
   /*strcpy(fs->request.payload.fn_17_79.Entry.JobFileName,"!NWClient");*/
   strcpy(fs->request.payload.fn_17_79.JobTextDescription,"!NWClient spooled output");
   fs->request.payload.fn_17_79.ClientRecordArea.TabSize=Settings->Tabs;
   fs->request.payload.fn_17_79.ClientRecordArea.Copies=Settings->Copies;
   fs->request.payload.fn_17_79.ClientRecordArea.Lines=Settings->Lines;
   fs->request.payload.fn_17_79.ClientRecordArea.Columns=Settings->Columns;

   fs->request.payload.fn_17_79.ClientRecordArea.CtrlFlags=
      (Settings->FF?0:8)|(Settings->Notify?16:0)|(Settings->Tabs?0x40:0)|(Settings->Banner?0x80:0);

   if(Settings->Form)
      strcpy(fs->request.payload.fn_17_79.ClientRecordArea.FormName,Settings->Form);
   if(Settings->Banner)
      strcpy(fs->request.payload.fn_17_79.ClientRecordArea.BannerName,Settings->Banner);
   if(Settings->Name)
      strcpy(fs->request.payload.fn_17_79.ClientRecordArea.FnameBanner,Settings->Name);
   /*strcpy(fs->request.payload.fn_17_79.ClientRecordArea.FnameHeader,"text2");*/

   if((r=ncp_request(REQ_17"LW6l6b6bl3wb13b4l50b" "bb4W16b6z13b13b14b80b",
                     "w6l6b6bL3wb13b4l",NULL,fs))<0)
   {  free(file);
      return NULL; }

   file->Is.PrintJob.Job=nw->reply.payload.fn_17_79.Entry.JobNumber;
   file->Mount=NULL;
   *file->Name=0;
   file->Action=0;
   file->Flags=FILE_WRITABLE|FILE_PRINTJOB;
   file->CurrentSize=-1;
   file->DirEntNum=file->VolNumber=-1;

   memcpy(&handle[1],&nw->reply.payload.fn_17_79.Entry.JobFileHandle,4);
   handle[0]=handle[1]+1;  /* heavy magic here */
   memcpy(file->Handle,&handle,6);

   AddHead(&fs->Files,file);

   return file;
}


/* +---------------------------------------------------------------------+
   |                                                                     |
   |   Close an opened file on the file server.                          |
   |                                                                     |
   +---------------------------------------------------------------------+ */

int nwclient_close(struct File *file,register struct NWClient *nw)
{  int r;
   register struct Server *fs=file->Server;

   #if defined DEBUG
   Trace("\n nwclient_close");
   #endif

   if(file->Flags&FILE_PRINTJOB)
   {
   	fs->request.header.Function=0x17;
      fs->request.payload.fn_17_7f.Length=9;
      fs->request.payload.fn_17_7f.SubFunction=file->CurrentSize?0x7f:0x80;
      fs->request.payload.fn_17_7f.Queue=file->Is.PrintJob.Queue;
      fs->request.payload.fn_17_7f.Job=file->Is.PrintJob.Job;

      r=ncp_request(REQ_17"LL",NULL,NULL,fs);
   }
   else
   {
   	fs->request.header.Function=0x42;   /* close file */
      memcpy(fs->request.payload.fn_42.Handle,file->Handle,6);
   
      r=ncp_request("z6b",NULL,NULL,fs);
   }

   RemoveNode(file);
   free(file);

   #if defined DEBUG && (DEBUG&2)
   Trace("\n closed");
   #endif

	return r;
}


/* +---------------------------------------------------------------------+
   |                                                                     |
   |   Read from a file.                                                 |
   |                                                                     |
   +---------------------------------------------------------------------+ */

int nwclient_read(struct File *file,void *buffer,int offset,int length,struct NWClient *nw)
{  int r;
   register struct Server *fs=file->Server;

	#if defined DEBUG
   Trace("\n nwclient_read");
   #endif

	fs->request.header.Function=0x48;   /* read from file */
   memcpy(fs->request.payload.fn_48.Handle,file->Handle,6);
   fs->request.payload.fn_48.Offset=offset;
   fs->request.payload.fn_48.Length=length;
   
	if((r=ncp_request("z6bLW","W*b",NULL,fs))<0)
      return r;

   memcpy(buffer,&nw->reply.payload.fn_48.Data,nw->reply.payload.fn_48.Length);

   return nw->reply.payload.fn_48.Length;
}


/* +---------------------------------------------------------------------+
   |                                                                     |
   |   Write to a file.                                                  |
   |                                                                     |
   +---------------------------------------------------------------------+ */

int nwclient_write(struct File *file,const void *buffer,int offset,int length,register struct NWClient *nw)
{  int r;
   register struct Server *fs=file->Server;

	#if defined DEBUG
   Trace("\n nwclient_write");
   #endif

   fs->request.header.Function=0x49;  /* write to file */
   memcpy(fs->request.payload.fn_49.Handle,file->Handle,6);
   fs->request.payload.fn_49.Offset=offset;
   fs->request.payload.fn_49.Length=length;
   memcpy(&fs->request.payload.fn_49.Data,buffer,length);
   
   if((r=ncp_request("z6bLW*b",NULL,NULL,fs))<0)
      return r;

   if(length==0)
      file->CurrentSize=offset;

   return length;
}


/* +---------------------------------------------------------------------+
   |                                                                     |
   |   Calculate the size of a file.                                     |
   |                                                                     |
   +---------------------------------------------------------------------+ */

int nwclient_filesize(struct File *file,struct NWClient *nw)
{  int r;
   register struct Server *fs=file->Server;

   #if defined DEBUG
   Trace("\n nwclient_filesize");
   #endif

   fs->request.header.Function=0x47;  /* get file size */
   memcpy(fs->request.payload.fn_47.Handle,file->Handle,6);
   if((r=ncp_request("z6b","L",NULL,fs))<0)
      return 0;

   return nw->reply.payload.fn_47.Size;
}


/* ----- EOF FILEIO.C ----- */


/* ----- EOF fileio.c ----- */

